<?php

/**
 * Image Utility Service
 * =====================
 * Service for infrequently used image utilities
 * =============================================
 *
 * @category   ExpressionEngine Add-on
 * @package    JCOGS Image
 * @author     JCOGS Design <contact@jcogs.net>
 * @copyright  Copyright (c) 2021 - 2023 JCOGS Design
 * @license    https://jcogs.net/add-ons/jcogs_img/license.html
 * @version    1.3.21.1
 * @link       https://JCOGS.net/
 * @since      File available since Release 1.2.12
 */

namespace JCOGSDesign\Jcogs_img\Service;

require_once PATH_THIRD . "jcogs_img/vendor/autoload.php";
require_once PATH_THIRD . "jcogs_img/config.php";

use enshrined\svgSanitize\Sanitizer;
use Imagine\Factory;
use Imagine\Image;
use Imagine\Image\Palette;
use Imagine\Image\Point;
use FilesystemIterator;
use JCOGSDesign\Jcogs_img\Library\HaarDetector;

class ImageUtilities
{
    public $settings;
    public static $browser_image_format_support;
    public static $valid_server_image_formats;
    public static $license_status;
    public static $current_params;
    private $valid_params;
    private $dimension_params;

    public function __construct()
    {
        ee()->load->helper('string');
        $this->settings = ee('jcogs_img:Settings')::$settings;
        $this->valid_params = [
            // Params mix of CE-Image and Intervention options
            'add_dims' => null,
            // 'allow_overwrite_original' => '',
            'allow_scale_larger' => $this->settings['img_cp_allow_scale_larger_default'],
            // 'ascii_art' => '',
            'attributes' => null,
            // 'auto_cache' => null,
            'bg_color' => $this->settings['img_cp_default_bg_color'],
            // 'bg_color_default' => $this->settings['img_cp_default_bg_color'],
            'border' => null,
            // 'cache' => '',
            'cache_dir' => $this->settings['img_cp_default_cache_directory'],
            'consolidate_class_style' => $this->settings['img_cp_class_consolidation_default'],
            'create_tag' => '',
            'crop' => 'n|center,center|0,0|y|3', // crop:yes_or_no|position|offset|smart_scale|face_sensitivity
            'default_img_width' => $this->settings['img_cp_default_img_width'],
            'default_img_height' => $this->settings['img_cp_default_img_height'],
            // 'debug' => '',
            // 'dir_permissions' => '',
            // 'disable_xss_check' => 'no',
            'encode_urls' => 'yes',
            'exclude_regex' => null,
            'fallback_src' => null,
            'filename' => null,
            'filename_prefix' => null,
            'filename_suffix' => null,
            'filter' => null,
            'flip' => null,
            // 'force_remote' => 'no',
            'hash_filename' => 'no',
            // 'hide_relative_path' => 'no',
            // 'image_opt' => 'no,lossless,png',
            'interlace' => 'no',
            // 'manipulate' => 'yes',
            'output' => '',
            'overwrite_cache' => 'no',
            // 'parse' => '',
            'quality' => $this->settings['img_cp_jpg_default_quality'],
            'reflection' => null,
            // 'refresh' => '',
            // 'remote_cache_time' => '-1',
            // 'remote_dir' => '',
            'rotate' => null,
            'rounded_corners' => '',
            'save_type' => null,
            'src' => 'not_set',
            'text' => null,
            // 'top_colors' => '',
            // 'unique' => '',
            'url_only' => '',
            'watermark' => null,
            // Non CE-Image parameters
            'add_dimensions' => null,
            'aspect_ratio' => null,
            'auto_sharpen' => $this->settings['img_cp_enable_auto_sharpen'],
            'bulk_tag' => 'n',
            'cache' => $this->settings['img_cp_default_cache_duration'],
            'cache_mode' => 'f',
            'disable_browser_checks' => null,
            'face_crop_margin' => 0,
            'face_detect_sensitivity' => 3,
            'fit' => 'contain',
            // 'image_permissions' => '',
            'image_path_prefix' => $this->settings['img_cp_path_prefix'],
            'lazy' => '',
            'palette_size' => 8,
            'png_quality' => $this->settings['img_cp_png_default_quality'],
            'save_as' => null,
            'srcset' => null,
            'sizes' => null,
            'svg_passthrough' => $this->settings['img_cp_enable_svg_passthrough'],
            'svg_width' => $this->settings['img_cp_default_img_width'],
            'svg_height' => $this->settings['img_cp_default_img_height'],
            'use_image_path_prefix' => null,
        ];
        $this->dimension_params = [
            'width' => null,
            'height' => null,
            'max' => '',
            'max_height' => '',
            'max_width' => '',
            'min' => '',
            'min_height' => '',
            'min_width' => '',
        ];
        $this::$current_params = [];
    }

    /**
     * Calculate anti-aliased value for pixel during combination of two images
     * https://www.exorithm.com/exorithm-execute-algorithm-view-run-algorithm-antialias_pixel/
     * Modified by JCOGS Design for use as a general mask algorithm
     *
     * @param  object|resource $image
     * @param  int $x // X of pixel to be anti-aliased
     * @param  int $y // Y of pixel to be anti-aliased
     * @param  int $colour // Anti-aliased pixel colour (default transparent)
     * @return object|resource $image
     */
    public function antialias_pixel(object $image, int $x, int $y, int $colour = null)
    {

        // Check that X/Y within bounds of image passed
        if ($x >= imagesx($image) || $x < 0 || $y >= imagesy($image) || $y < 0) {
            return $image;
        }

        // If colour not set, use the colour of current pixel
        if (!$colour) {
            $colour = imagecolorat($image, (int) $x, (int) $y);
        }

        // Get the colour we are going to set pixel to 
        $c = imagecolorsforindex($image, $colour);
        $r = $c['red'];
        $g = $c['green'];
        $b = $c['blue'];
        $t = $c['alpha'];

        // Get transparency of existing pixel
        $transparency = (imagecolorat($image, $x, $y) & 0x7F000000) >> 24;


        // Get average opacity of surrounding 9 pixels (or whatever number is available)
        $opacity_count = 0;
        $pixel_count = 0;
        for ($i = 0; $i < 3; $i++) {
            for ($j = 0; $j < 3; $j++) {
                if ($x + $i - 1 >= 0 && $y + $j - 1 >= 0 && $x + $i - 1 < imagesx($image) && $y + $j - 1 < imagesy($image))
                    // If scaled image pixel is opaque add 1 to opacity_count
                    $opacity_count += (imagecolorat($image, $x + $i - 1, $y + $j - 1) & 0x7F000000) >> 24;
                $pixel_count++;
            }
        }
        // Average opacity - 1 if all pixels opaque, 0 if all transparent
        $average_opacity = $opacity_count / $pixel_count;


        // // Adjust opacity by 1-$average_opacity
        // imagesetpixel($image,$x,$y,imagecolorallocatealpha($image, $r, $g, $b, $average_opacity));

        // Apply target colour with transparency of existing pixel
        imagesetpixel($image, $x, $y, imagecolorallocatealpha($image, $r, $g, $b, round(127 - $average_opacity / 8, 0)));

        return $image;
    }

    /**
     * Utility function: Scans cache folder and deletes images that have expired
     *
     * @param  bool $force - run an audit even if one is not due / required
     * @return mixed
     */
    public function cache_audit(bool $force = false)
    {
        // Is cache_auditing enabled, and auto-cache interval not 0?
        if (substr(strtolower($this->settings['img_cp_enable_cache_audit']), 0, 1) != 'y' || !($this->settings['img_cp_default_cache_audit_after'] > 0)) {
            return;
        }

        // Do we have a marker from last audit?
        $marker = ee('jcogs_img:Utilities')->cache_utility('get', '/' . JCOGS_IMG_CLASS . '/' . 'image_cache_audit');
        $last_audit = $marker ? $marker : 0;

        // Are we due for an audit?
        if ($force || time() - $last_audit > $this->settings['img_cp_default_cache_audit_after']) {
            // Audit is due if gap between last audit and now is greater than the cache audit delay
            
            // Create a container for files found and marked valid
            $cache_status = [];

            // Create a counter for files removed
            $files_removed = 0;
            
            // Get a copy of the current cache log - clean that up at same time ... 
            $cache_log = ee('jcogs_img:Utilities')->cache_utility('get', JCOGS_IMG_CLASS . '/cache_log');
            
            // Scan log file to get list of unique cache locations
            $cache_locations = [];
            if ($cache_log) {
                foreach ($cache_log as $key => $value) {
                    if (is_array($value) && array_key_exists('cache_dir', $value)) {
                        array_key_exists($value['cache_dir'], $cache_locations) ? $cache_locations[$value['cache_dir']] += 1 : $cache_locations[$value['cache_dir']] = 1;
                    }
                }

            ee('jcogs_img:Utilities')->debug_message(sprintf(lang('jcogs_img_cache_audit_begin'),count($cache_locations)),$cache_locations);

            // Now scan each cache location... 
                foreach ($cache_log as $cache_fragment => $array) {
                    // Ignore any entry that is not an array value
                    if (is_array($array) && array_key_exists('cache_dir', $array)) {
                        // Get the appropriate file path
                        $this_cache_path = $array['cache_dir'] != '' ? $array['cache_dir'] : $this->settings['img_cp_default_cache_directory'];
                        $this_file_path = rtrim(ee('jcogs_img:Utilities')->path($this_cache_path . '/' . $cache_fragment), '/');
                        // is it still there?
                        $age_of_file = @filemtime($this_file_path);
                        // how long was it supposed to be cached for?
                        $cache_duration_when_saved = $this->get_file_cache_duration($this_file_path);
                        // check to see if cache timer has expired
                        $valid =  $cache_duration_when_saved == -1 || ($this->settings['img_cp_default_cache_duration'] > 0 && $cache_duration_when_saved && time() - $age_of_file < $cache_duration_when_saved);
                        if (!$valid) {
                            // Not valid so delete the file and clear cache entry
                            if (file_exists($this_file_path)) {
                                unlink($this_file_path);
                                $files_removed++;
                            }
                            if ($cache_log && isset($cache_log[$cache_fragment])) {
                                unset($cache_log[$cache_fragment]);
                            }
                            // And remove from $cache_status
                            if (isset($cache_status[$cache_fragment])) {
                                unset($cache_status[$cache_fragment]);
                            }
                        } else {
                            // mark cache status entry as valid
                            $cache_status[$array['cache_dir']][$cache_fragment] = 'valid';
                        }
                    }
                }

                // Update cache log with Changes
                ee('jcogs_img:Utilities')->cache_utility('save', JCOGS_IMG_CLASS . '/cache_log', $cache_log, 31536000);
                
                // Now scan any remaining files in the cache directories found: we know any found are not in log
                foreach ($cache_locations as $cache => $count) {
                    // If cache folder exists...
                    if (ee('Filesystem')->exists(ee('jcogs_img:Utilities')->path($cache))) {
                        // ... get a list of all files in image cache folders and audit
                        // $file_list = ee('Filesystem')->getDirectoryContents(ee('jcogs_img:Utilities')->path($cache));
                        $file_list = [];
                        $iterator = new FilesystemIterator(ee('jcogs_img:Utilities')->path($cache));
                        foreach ($iterator as $fileinfo) {
                            if ($fileinfo->getType() == 'file') {
                                $file_list[] = $fileinfo->getPathname();
                            }
                        }
                        
                        // ... loop through this list
                        foreach ($file_list as $file) {
                            if ((!array_key_exists($cache, $cache_status) || !array_key_exists(pathinfo($file)['basename'], $cache_status[$cache])) && ee('Filesystem')->exists('/' . $file)) {
                                // ... for each file found, if there is no cache status file for this directory or if file is not in cache status file then audit it
                                $age_of_file = filemtime('/' . $file);
                                // how long was it supposed to be cached for?
                                $cache_duration_when_saved = $this->get_file_cache_duration('/' . $file);
                                // check to see if cache timer has expired
                                $valid =  $cache_duration_when_saved == -1 || ($this->settings['img_cp_default_cache_duration'] > 0 && $cache_duration_when_saved && time() - $age_of_file < $cache_duration_when_saved);
                                if (!$valid) {
                                    // Not valid so delete the file and clear cache entry
                                    @unlink('/' . $file);
                                    $files_removed++;
                                }
                            }
                        }
                    }
                }
                // Update cache audit flag
                ee('jcogs_img:Utilities')->cache_utility('save', JCOGS_IMG_CLASS . '/' . 'image_cache_audit', time(), $this->settings['img_cp_default_cache_audit_after']);
                ee('jcogs_img:Utilities')->debug_message(sprintf(lang('jcogs_img_cache_audit_end'),$files_removed));
            }    
        }
        if ($force) {
            return ['locations' => count($cache_locations), 'removed' => $files_removed];
        } else {
            return false;
        }
    }

    /**
     * Converts a GD Image object to Imagine/Image object
     *
     * @param  resource|object $gdimage
     * @return resource|object|bool
     */
    public function convert_GDImage_object_to_image($gdimage)
    {
        // Make sure we got something 
        if (!$gdimage) {
            return false;
        }

        try {
            $imagine = (new Factory\ClassFactory())->createImage(
                Factory\ClassFactoryInterface::HANDLE_GD,
                $gdimage,
                new Palette\RGB(),
                new Image\Metadata\MetadataBag(),
            );
        } catch (\Imagine\Exception\RuntimeException $e) {
            // Creation of image failed.
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_imagine_error'), $e->getMessage());
            return false;
        }
        return $imagine;
    }

    /**
     * Converts an image (on $path) to a GDImage object
     *
     * @param  string $type
     * @param  string $path
     * @return object|resource|bool
     */
    public function convert_image_to_GDImage_object(string $type, string $path)
    {
        // Finally change $content->source_image_raw to a GdImage entity
        switch (strtolower($type)) {
            case 'bmp':
                $return_image = imagecreatefrombmp($path);
                break;
            case 'gif':
                $return_image = imagecreatefromgif($path);
                break;
            case 'jpeg':
                $return_image = imagecreatefromjpeg($path);
                break;
            case 'jpg':
                $return_image = imagecreatefromjpeg($path);
                break;
            case 'png':
                $return_image = imagecreatefrompng($path);
                break;
            case 'wbmp':
                $return_image = imagecreatefromwbmp($path);
                break;
            case 'webp':
                $return_image = imagecreatefromwebp($path);
                break;
            case 'xbm':
                $return_image = imagecreatefromxbm($path);
                break;
            case 'xpm':
                $return_image = imagecreatefromxpm($path);
                break;
        }
        if (!$return_image) {
            return false;
        }
        return $return_image;
    }

    /**
     * Utility function: Converts rgb or rgba into GdImage three / four digit colour
     *
     * @param string $colour_string
     * @return array|bool $colours
     */
    public function convert_rgba_to_GdImage_format(string $colour_string)
    {
        if (preg_match('/rgba\((.*)\)/', $colour_string, $matches)) {
            $values = explode(',', $matches[1]);
            $values[3] = (int)((1 - $values[3]) * 127);
            return $values;
        } elseif (preg_match('/rgb\((.*)\)/', $colour_string, $matches)) {
            $values = explode(',', $matches[1]);
            // set opacity value to 1 to imply opaque and so return four values
            $values[3] = 127;
            return $values;
        } else {
            return false;
        }
    }

    /**
     * Utility function: Converts rgb or rgba into Imagine Color Palette
     *
     * @param string $colour_string
     * @return object|bool $colours
     */
    public function convert_rgba_to_Imagine_RGB(string $colour_string)
    {
        if (preg_match('/rgba\((.*)\)/', $colour_string, $matches)) {
            $values = explode(',', $matches[1]);
            $values[3] = (int)($values[3] * 100);
        } elseif (preg_match('/rgb\((.*)\)/', $colour_string, $matches)) {
            $values = explode(',', $matches[1]);
            // set opacity value to 1 to imply opaque and so return four values
            $values[3] = 100;
        } else {
            return false;
        }
        $colors = [[(int) $values[0], (int) $values[1], (int) $values[2]], (int) $values[3]];

        return (new Palette\RGB())->color(...$colors);
    }

    /**
     * Returns a semi-transparent image to overlay on images when in demo mode
     * The image is saved as a base64 encoded png file, without the mime type and base 64 header
     * added in to make it into a Data URI format image (i.e. 'data:image/png;base64,$imagedata')
     *
     * @return string
     */
    public function demo_image()
    {
        return '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 ';
    }

    /**
     * Checks image content to see if it is an HEIC image format
     * Uses method from php-heic-to-jpg from here https://github.com/MaestroError/php-heic-to-jpg
     * @param  string|null $image
     * @return bool
     */
    public function detect_heic(?string $image = null)
    {

        if (!$image || !is_string($image))
            return false;

        $magicNumber = strtolower(trim(substr(substr($image, 0, 12), 8)));

        $heicMagicNumbers = [
            'heic', // official
            'mif1', // unofficial but can be found in the wild
            'ftyp', // 10bit images, or anything that uses h265 with range extension
            'hevc', // brands for image sequences
            'hevx', // brands for image sequences
            'heim', // multiview
            'heis', // scalable
            'hevm', // multiview sequence
            'hevs', // multiview sequence
        ];

        return in_array($magicNumber, $heicMagicNumbers);
    }

    /**
     * Utility function: Detect version of PNG file
     * PNG8 and PNG24 files can cause problems with GdImage imagemerge operations due
     * lack of a distinct transparency layer. This function works out what format a PNG
     * file is so appropriate steps can be taken.
     * Code from https://stackoverflow.com/a/57547867/6475781
     * Modified to work with image string rather than file
     *
     * @param string $image - a string version of image file
     * @return array|bool
     */
    public function detect_png_version(string $image)
    {
        if (substr($image, 0, 4) !== chr(0x89) . 'PNG') {
            // This is not a PNG
            return false;
        }

        // Make sure we have an IHDR
        if (substr($image, 12, 4) !== 'IHDR') {
            return false;
        }

        // PNG actually stores Width and height integers in big-endian.
        $width = unpack('N', substr($image, 16, 4))[1];
        $height = unpack('N', substr($image, 20, 4))[1];

        // Bit depth: 1 byte
        // Bit depth is a single-byte integer giving the number of bits per sample or
        // per palette index (not per pixel).
        //
        // Valid values are 1, 2, 4, 8, and 16, although not all values are allowed for all color types.
        $bitDepth = ord(substr($image, 24, 1));

        // Pixel format
        // https://en.wikipedia.org/wiki/Portable_Network_Graphics#Pixel_format

        // Color type is a single-byte integer that describes the interpretation of the image data.
        // Color type codes represent sums of the following values:
        // 1 (palette used), 2 (color used), and 4 (alpha channel used).
        //
        // Valid values are 0, 2, 3, 4, and 6.
        $colorType = ord(substr($image, 25, 1));

        $colorTypes = [
            0 => 'Greyscale',
            2 => 'Truecolour',
            3 => 'Indexed-colour',
            4 => 'Greyscale with alpha',
            6 => 'Truecolour with alpha',
        ];

        $colorTypeText = $colorTypes[$colorType];

        $pngType = '?';
        // If the bitdepth is 8 and the colortype is 3 (Indexed-colour) you have a PNG8
        if ($bitDepth === 8 && $colorType === 3) {
            $pngType = 'PNG8';
        }

        // If the bitdepth is 8 and colortype is 2 (Truecolour) you have a PNG24.
        if ($bitDepth === 8 && $colorType === 2) {
            $pngType = 'PNG24';
        }

        // If the bitdepth is 8 and colortype is 6 (Truecolour with alpha) you have a PNG32.
        if ($bitDepth === 8 && $colorType === 6) {
            $pngType = 'PNG32';
        }

        return [
            'width'         => $width,
            'height'        => $height,
            'bit-depth'     => $bitDepth,
            'colorType'     => $colorType,
            'colorTypeText' => $colorTypeText,
            'pngType'       => $pngType
        ];
    }

    /**
     * Utility function: Detect and sanitize SVG images
     * This will either return a sanitized SVG/XML string or bool false 
     * if XML parsing failed (usually due to a badly formatted file).
     * 
     * Uses SVG Sanitizer library from https://github.com/darylldoyle/svg-sanitizer
     *
     * @param string $image - a string version of image file
     * @return bool|string $image - a sanitized version of image file
     */
    public function detect_sanitize_svg(string $image)
    {
        if(empty($image)) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_SVG_missing'));
            return false;
        }

        // Create a new sanitizer instance
        $sanitizer = new Sanitizer();

        // Pass image to the sanitizer and get it back clean
        $cleanSVG = $sanitizer->sanitize($image);

        if ($cleanSVG) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_SVG_valid'));
        }

        return $cleanSVG;
    }

    /**
     * Generate points for rotated regular polygon
     *
     * @param int $x
     * @param int $y
     * @param int $radius
     * @param int $vertices (default 3)
     * @param int $rotation (default 0)
     * @return array
     */
    public function draw_rotated_polygon(int $x, int $y, int $radius, int $vertices = 3, int $rotation = 0)
    {
        // $x, $y -> Position in the image
        // $radius -> Radius of circle enclosing the polygon
        // $spikes -> Number of vertices

        $angle = 360 / $vertices;

        // Get the coordinates of the polygon
        $coordinates = array();
        for ($i = 0; $i < $vertices; $i++) {
            $coordinates[] = new Point((int) round($x + ($radius * cos(deg2rad(270 - $angle * $i + $rotation))), 0), (int) round($y + ($radius * sin(deg2rad(270 - $angle * $i + $rotation))), 0));
        }

        // Return the coordinates
        return $coordinates;
    }

    /**
     * Generate points for rotated star
     * With inspiration from examples on
     * http://www.php.net/manual/en/function.imagefilledpolygon.php
     *
     * @param int $x
     * @param int $y
     * @param int $radius
     * @param int $spikes
     * @param float $split
     * @param int $rotation
     * @return array
     */
    public function draw_rotated_star(int $x, int $y, int $radius, int $spikes = 5, float $split = 0.5, int $rotation = 0)
    {
        // $x, $y -> Position in the image
        // $radius -> Radius of the star
        // $spikes -> Number of spikes

        $coordinates = array();
        $angle = 360 / $spikes;

        // Get the coordinates of the outer shape of the star
        $outer_shape = array();
        for ($i = 0; $i < $spikes; $i++) {
            $outer_shape[$i]['x'] = (int) round($x + ($radius * cos(deg2rad(270 - $angle * $i + $rotation))), 0);
            $outer_shape[$i]['y'] = (int) round($y + ($radius * sin(deg2rad(270 - $angle * $i + $rotation))), 0);
        }

        // Get the coordinates of the inner shape of the star
        $inner_shape = array();
        for ($i = 0; $i < $spikes; $i++) {
            $inner_shape[$i]['x'] = (int) round($x + ($split * $radius * cos(deg2rad(270 - 180 - $angle * $i + $rotation))), 0);
            $inner_shape[$i]['y'] = (int) round($y + ($split * $radius * sin(deg2rad(270 - 180 - $angle * $i + $rotation))), 0);
        }

        // Bring the coordinates in the right order
        foreach ($inner_shape as $key => $value) {
            if ($key == (floor($spikes / 2) + 1))
                break;
            $inner_shape[] = $value;
            unset($inner_shape[$key]);
        }

        // Reset the keys
        $i = 0;
        foreach ($inner_shape as $value) {
            $inner_shape[$i] = $value;
            $i++;
        }

        // "Merge" outer and inner shape
        foreach ($outer_shape as $key => $value) {

            $coordinates[] = new Point($outer_shape[$key]['x'], $outer_shape[$key]['y']);
            $coordinates[] = new Point($inner_shape[$key]['x'], $inner_shape[$key]['y']);
        }

        // Return the coordinates
        return $coordinates;
    }

    /**
     * Processes and returns encoded image as data-url string
     *
     * @param  object|null $image
     * @param  string|null $format
     * @param  string|null $quality
     * @return string
     */
    public function encode_base64(object $image = null, $format = null, $quality = null)
    {
        if (is_null($image) || is_null($format) || is_null($quality)) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_encode_too_few_params'));
            return '';
        }
        return sprintf(
            'data:%s;base64,%s',
            $format,
            base64_encode((string) $image->get($format, ['quality' => $quality]))
        );
    }

    /**
     * Face detection - heavily based on HAARPHP example code.
     * https://github.com/foo123/HAARPHP
     * Returns an array of arrays - 
     *  - first entry gives x y width height of bounding box
     *  - each subsequent entry gives x, y, width, height of detected faces
     *
     * @param  object|null $image
     * @param  int         $sensitivity
     * @param  array|null  $cascade
     * @return mixed
     */
    public function face_detection($image = null, int $sensitivity = 3, array $cascade = null)
    {
        ee('jcogs_img:Utilities')->debug_message(sprintf(lang('jcogs_img_face_detect_start'), $sensitivity));
        // Start a timer for this operation run
        $time_start = microtime(true);

        if (is_null($image)) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_face_detect_too_few_params'));
            return '';
        }

        if (is_null($cascade)) {
            require PATH_THIRD . "jcogs_img/Library/haarcascade_frontalface_alt.php";
            $cascade = $haarcascade_frontalface_alt;
        }

        $sensitivity = min(max(1, $sensitivity), 9); // Normalise value to range 1-9
        $sensitivity = ($sensitivity - 3); // 3 == 0
        $scale = (5 + $sensitivity) / 15; // Normalise based on $sensitivity 1 => 20%, 9 => 93%.
        // Create a new detector
        $faceDetector = new HaarDetector($cascade);

        // Look for faces
        $found = $faceDetector
            // normalise image to some standard dimensions eg. 150 px width 
            // provides performance / accuracy trade-off
            // $sensitivity 
            ->image($image, $scale)
            ->cannyThreshold(array('low' => 80, 'high' => 200))
            ->detect(1, 1.1, 0.12, 1, 0.2, false);

        // if detected
        if ($found) {
            // $numFeatures = count($faceDetector->objects);
            $collection = array('min-x' => null, 'min-y' => null, 'max-x' => null, 'max-y' => null);
            // create array of summary found image from original image
            $detectedFaces = array_map(function ($face) use (&$collection) {
                $detectedFace = array(
                    'x' => $face->x,
                    'y' => $face->y,
                    'width' => $face->width,
                    'height' => $face->height
                );
                $collection['min-x'] = is_null($collection['min-x']) ? $face->x : min($face->x, $collection['min-x']);
                $collection['min-y'] = is_null($collection['min-y']) ? $face->y : min($face->y, $collection['min-y']);
                $collection['max-x'] = is_null($collection['max-x']) ? $face->x + $face->width : max($face->x + $face->width, $collection['max-x']);
                $collection['max-y'] = is_null($collection['max-y']) ? $face->y + $face->height : max($face->y + $face->height, $collection['max-y']);
                return $detectedFace;
            }, $faceDetector->objects);
            // Write to log
            ee('jcogs_img:Utilities')->debug_message(sprintf(lang('jcogs_img_face_detect_ends'), microtime(true) - $time_start));
            // Work out width of containing shape and insert into return matrix
            return array_merge([[
                'x' => $collection['min-x'],
                'y' => $collection['min-y'],
                'width' => $collection['max-x'] - $collection['min-x'],
                'height' => $collection['max-y'] - $collection['min-y']
                ]], $detectedFaces);
        }
        ee('jcogs_img:Utilities')->debug_message(sprintf(lang('jcogs_img_face_detect_none_found'), microtime(true) - $time_start));
        return false;
    }


    /**
     * Utility function: Tries to get a local copy of an image from a path
     * Updated Oct 2022 - in cases where base_url is not set to server web root, do some 
     * stuff to avoid getting confused about where files are locally.
     *
     * @param string $path
     * @return array|bool
     */

    public function get_a_local_copy_of_image(string $path)
    {
        // Clean up the path just in case
        if (!$path = ee('Security/XSS')->clean($path)) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_XSS_fail'), [$path]);
            return false;
        };

        // Get some info about where image is
        $parse_src = parse_url($path);
        $base_url = ee()->config->item('base_url');

        // Is file link to this domain (i.e. a local file?) 
        // Test sequence: 
        // 1) is host set? If no, probably a local file path
        // 2) does $path begin with same string as base_url? If so, local file is remainder
        // 3) does $path begin site_url? If so, probably a local file is [path]

        if (
            !isset($parse_src['host']) ||
            strpos($path, $base_url) === 0 ||
            (isset($parse_src['host']) && $parse_src['host'] == parse_url($base_url)['host'])
        ) {
            // Seems to be a local file... 
            if (!isset($parse_src['host'])) {
                // No Host defined)
                // Strip leading '/' if one set and generate local path to test
                $path = ee('jcogs_img:Utilities')->path(substr($parse_src['path'], 0, 1) == '/' ? substr($parse_src['path'], 1) : $parse_src['path']);
            } elseif (strpos($path, $base_url) === 0) {
                // URL appears to be from this site
                // Strip off base URL form start of string
                $path = ee('jcogs_img:Utilities')->path(parse_url(str_replace($base_url, '', $path))['path']);
            } elseif (isset($parse_src['host']) && $parse_src['host'] == parse_url(base_url())['host']) {
                // Strip leading '/' if one set and generate local path to test
                $path = ee('jcogs_img:Utilities')->path(substr($parse_src['path'], 0, 1) == '/' ? substr($parse_src['path'], 1) : $parse_src['path']);
            }

            // Strip off a trailing '/' if there is one... 
            $path = rtrim($path, '/');

            if (!ee('Filesystem')->exists($path)) { // the ee call is for maximum compatibility with its Flysystem file system.
                // Try again with decoded path (see if this is a wygwam sourced image...)
                $path = urldecode($path);
                if (!ee('Filesystem')->exists($path)) {
                    // The path given really doesn't appear to exist
                    // So fall back to using basic php test (which may cause problems on Flysystem using systems)
                    ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_unable_to_open_path_retry'), $path);;
                    if (!is_readable($path)) {
                        // try one last time
                        ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_unable_to_open_path_2'), $path);
                        return false;
                    }
                }
            }

            // If we get here the path to file exists (huzzah!) - now see if we can get content from it
            // Try to get a copy of the image
            $image_source = $this->get_file_from_local($path);
            if (!$image_source) {
                // we have got a junk path so return
                return false;
            }
        } else {
            // Try and get image from remote URL
            if (!$image_source = $this->get_file_from_remote($path)) {
                // Last effort - see if file is in CE Image Remote cache (if there is one)
                if (!pathinfo($path)['basename']) {
                    return false;
                }
                if (!$path = $this->look_for_ce_image_remote_files(pathinfo($path)['basename'])) {
                    // We got nothing so bale...)
                    return false;
                }
                // We got something! So set that to our image and continue ... 
                $image_source = $this->get_file_from_local($path);
                if (!$image_source) {
                    // if we get nothing bale
                    return false;
                }
            }
        }
        return ['image_source' => $image_source, 'path' => $path];
    }

    /**
     * Utility function: Checks to see what image formats browser will accept
     * Browser accept: responses only list novel formats (webp, avif etc.)
     * Uses two methods. 
     * First it tries the HTTP Accept header from $_SERVER superglobal 
     * This is google approved method 
     * (https://developers.google.com/speed/webp/faq#server-side_content_negotiation_via_accept_headers)
     * Then, since Apple doesn't always put this info into the HTTP_ACCEPT header for Safari also need to
     * check via the HTTP User Agent string.
     * Due to Chrome sometimes reporting itself as Safari, put Apple last so if anything else can match
     * it will etc.
     * 
     * Based on https://caniuse.com/?search=webp we need to filter for Versions less than the following:
     * Android Browser - 4.2
     * Chrome - 32
     * Edge - 18
     * Firefox - 65
     * Opera - 19
     * IE - any version
     * 
     * Reference UserAgent strings from https://myip.ms
     *
     * @return	array	array with parameters
     */
    private function get_browser_image_capabilities()
    {
        // Have we done this already on this instance?
        if (self::$browser_image_format_support) {
            return self::$browser_image_format_support;
        }

        // The HTTP_ACCEPT response header only lists additional formats, 
        // So start format list with commonly accepted formats
        $valid_browser_formats_base = [
            'jpg',
            'jpeg',
            'png',
            'gif',
            'bmp',
        ];

        // Set the capabilities to the default set for now
        self::$browser_image_format_support = $valid_browser_formats_base;

        // First get the Accept Header and see if we can decide from that.
        if (isset($_SERVER['HTTP_ACCEPT']) && $accept_response = $_SERVER['HTTP_ACCEPT']) {
            // Unpack cited image formats (image/*)
            preg_match_all('/image\/(.*?),/', $accept_response, $matches);
            // Did we get any information?
            if (isset($matches) && count($matches[1]) > 0) {
                // Merge any we got into valid format list
                self::$browser_image_format_support = array_merge(self::$browser_image_format_support, $matches[1]);
            }
        }

        // See if we got something from Accept test, if not now check User Agent string
        if (count(self::$browser_image_format_support) == count($valid_browser_formats_base) && isset($_SERVER['HTTP_USER_AGENT'])) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_browser_user_agent_string'), [$_SERVER['HTTP_USER_AGENT']]);

            // Android 4 or less
            if (preg_match('/Android\s(.*);/', $_SERVER['HTTP_USER_AGENT'], $matches)) {
                ee('jcogs_img:Utilities')->debug_message(lang('android'), $matches);
                // True if second number greater than first... 
                if (version_compare('4.1', strval($matches[1]), 'lt')) {
                    array_push(self::$browser_image_format_support, 'webp');
                }
            } elseif (preg_match('/Chrome\/(.*)\s/', $_SERVER['HTTP_USER_AGENT'], $matches)) {
                ee('jcogs_img:Utilities')->debug_message(lang('chrome'), $matches);
                // Chrome 32 or less (also covers off Edge, Opera)
                // True if second number greater than first... 
                if (version_compare('31', strval($matches[1]), 'lt')) {
                    array_push(self::$browser_image_format_support, 'webp');
                }
            } elseif (preg_match('/Firefox\/(.*)$/', $_SERVER['HTTP_USER_AGENT'], $matches)) {
                ee('jcogs_img:Utilities')->debug_message(lang('firefox'), $matches);
                // Firefox 65 or less
                // True if second number greater than first... 
                if (version_compare('64', strval($matches[1]), 'lt')) {
                    array_push(self::$browser_image_format_support, 'webp');
                }
            } elseif (preg_match('/Version\/(.*)\sSafari/', $_SERVER['HTTP_USER_AGENT'], $matches)) {
                ee('jcogs_img:Utilities')->debug_message(lang('safari'), $matches);
                // Safari 16 or less
                // True if second number greater than first... 
                if (version_compare('16', strval($matches[1]), 'lt')) {
                    array_push(self::$browser_image_format_support, 'webp');
                }
            }
        }

        // Return the capability found
        ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_browser_image_support'), self::$browser_image_format_support);
        return self::$browser_image_format_support;
    }

    /**
     * Utility function: Read filename cache tag
     *
     * @param string $image_filename
     * @param int|null $default_cache_duration
     * @return int|bool $duration
     */
    public function get_file_cache_duration(string $image_filename, int $default_cache_duration = null)
    {
        $default_cache_duration = $default_cache_duration ?: $this->settings['img_cp_default_cache_duration'];
        $cache_duration_tag = explode($this->settings['img_cp_default_filename_separator'], $image_filename);
        if (count($cache_duration_tag)) {
            // Start from last element found and look for first one that looks like a cache duration... 
            for ($i = count($cache_duration_tag) - 1; $i >= 0; $i--) {
                if (isset($cache_duration_tag[$i]) && ctype_xdigit($cache_duration_tag[$i])) {
                    $cache_duration_when_saved = $cache_duration_tag[$i] == 'abcdef' ? -1 : hexdec($cache_duration_tag[$i]);
                    break;
                }
            }
        }
        if (!isset($cache_duration_when_saved)) {
            $cache_duration_when_saved = $default_cache_duration;
        }
        return is_int($cache_duration_when_saved) ? $cache_duration_when_saved : false;
    }

    /**
     * Utility function: Get a remote file using CURL
     * Returns either the file or false
     *
     * @param string $path
     * @return string|bool
     */
    private function get_file_contents_curl(string $path)
    {
        if (!$path) {
            // No path
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_gfcc_no_path'), $path);
            return false;
        }
        if (!function_exists('curl_init')) {
            // No CURL
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_gfcc_no_curl'), $path);
            return false;
        }

        // Set up CURL transaction

        // Thanks to ... https://stackanswers.net/questions/file-get-contents-fails-via-php-works-via-browser
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_URL, $path);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        // curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);

        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 2);
        curl_setopt($ch, CURLOPT_TIMEOUT, 5);
        curl_setopt($ch, CURLOPT_USERAGENT, $this->settings['img_cp_default_user_agent_string']);
        // curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 6.1; WOW64; rv:5.0) Gecko/20100101');

        // if either of these options are enabled, need this to skip an error
        if (!ini_get('open_basedir') && !ini_get('safe_mode')) {
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        }

        // Get the image if you can
        $remote_file = curl_exec($ch);

        // Check we got a good response (i.e. not 400, 500 or whatever)
        $https_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        // Clean up
        curl_close($ch);

        if($https_code != 200 || !$remote_file) {
            // Something went wrong ... 
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_gfcc_failed'), $path);
            return false;
        }

        // We probably got something ... 
        ee('jcogs_img:Utilities')->debug_message(lang('jcogs_gfcc_success'), $path);

        return $remote_file;
    }

    /**
     * Utility function: Get a file from local source
     * Uses file_get_contents, but if that fails tries two alternatives:
     * - less secure file_get_contents
     * - CURL
     * Returns the file or false
     *
     * @param string $path
     * @return string|bool
     */
    private function get_file_from_local(string $path)
    {

        ee('jcogs_img:Utilities')->debug_message(lang('jcogs_gffl_started'), $path);
        // Try first using file_get_contents, which works most of the time

        $local_file = false;

        $local_file = @file_get_contents($path);
        if (!$local_file) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_unable_to_open_path_1'), $path);;
            return false;
        }
        return $local_file;
    }

    /**
     * Utility function: Get a file from remote source
     * Uses file_get_contents, but if that fails tries two alternatives:
     * - less secure file_get_contents
     * - CURL
     * Returns the file or false
     *
     * @param string $path
     * @return string|bool
     */
    private function get_file_from_remote(string $path)
    {

        // Do some housekeeping to encourage file transfer can happen on difficult hosts
        @ini_set('allow_url_fopen', true);

        // No point trying if we can't access remote files... 
        if (!ee('jcogs_img:Licensing')->allow_url_fopen_enabled()) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_allow_url_fopen_disabled'));
            return false;
        }

        ee('jcogs_img:Utilities')->debug_message(lang('jcogs_gffr_started'), $path);
        // Try first using file_get_contents, which works most of the time

        // Do some connection massaging... 
        $options = array(
            'http' => array(
                'method' => "GET",
                'header' => "Accept-language: en\r\n",
                'timeout' => $this->settings['img_cp_default_php_remote_connect_time'],
                'user_agent' => $this->settings['img_cp_default_user_agent_string']
            )
        );
        $context = stream_context_create($options);

        $remote_file = false;
        $remote_path_array = pathinfo($path);
        $clean_remote_path = $remote_path_array['dirname'] . '/' . urlencode($remote_path_array['filename']);
        $clean_remote_path .= isset($remote_path_array['extension']) ? '.' . $remote_path_array['extension'] : '';

        $remote_file = @file_get_contents($clean_remote_path, false, $context);
        if (!$remote_file) {
            // No joy? 
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_unable_to_open_path_4'), $path);
            // Have a final go using CURL ...
            $remote_file = $this->get_file_contents_curl($path);
            if (!$remote_file) {
                // unable to read file from remote location
                return false;
            }
        }
        // ee('jcogs_img:Utilities')->debug_message(lang('jcogs_gffr_ending'),$path);
        return $remote_file;
    }
    
    /**
     * Utility function: get path to processed image directory
     * path prefix can either be the local site url (for full URL output) or some other arbitrary prefix (for CDNs etc.)
     *
     * @return string
     */
    public function get_image_path_prefix()
    {
        $image_path_prefix = '';
        // Has user requested "Full URL" output? If so they can't also have an arbitrary prefix.
        if (strtolower(substr($this->settings['img_cp_class_always_output_full_urls'], 0, 1)) == 'y') {
            $image_path_prefix = rtrim(base_url(), '/');
        } elseif (ee('jcogs_img:ImageUtilities')->get_parameters('image_path_prefix') && ee('jcogs_img:ImageUtilities')->get_parameters('use_image_path_prefix') && strtolower(substr(ee('jcogs_img:ImageUtilities')->get_parameters('use_image_path_prefix'), 0, 1)) == 'y') {
            // We have a prefix specified and is use of it requested?
            // Make sure prefix has a trailing '/'
            $image_path_prefix = rtrim(ee('jcogs_img:ImageUtilities')->get_parameters('image_path_prefix'), '/');
        }
        return $image_path_prefix;
    }

    /**
     * Utility function: Return mime type string for image based on save_as value
     * https://developer.mozilla.org/en-US/docs/Web/Media/Formats/Image_types
     * 
     * @param string $type
     * @return string
     */
    public function get_mime_type(string $type = null)
    {
        switch ($type) {
            case 'apng':
                return 'image/apng';
            case 'avif':
                return 'image/avif';
            case 'bmp':
                return 'image/bmp';
            case 'cur':
                return 'image/x-icon';
            case 'gif':
                return 'image/gif';
            case 'ico':
                return 'image/x-icon';
            case 'jpg':
            case 'jpeg':
                return 'image/jpeg';
            case 'png':
                return 'image/png';
            case 'svg':
                return 'image/svg+xml';
            case 'tif':
                return 'image/tiff';
            case 'tiff':
                return 'image/tiff';
            case 'wbmp':
                return 'image/vnd.wap.wbmp';
            case 'webp':
                return 'image/webp';
            case 'xbm':
                return 'image/xbm';
            case 'xpm':
                return 'image/xpm';
            default:
                ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_unknown_image_format'), $type);
                return '';
        }
    }

    /**
     * Utility function: If called without a parameter adds properties to $contents->params for
     * each valid parameter found in the tag, and loads specified value (or default)
     * for each of these valid parameters.
     * 
     * If called with a parameter returns the default value for the parameter, or false.
     * 
     * @param mixed $request_param
     * @param bool $get_default
     * @return string|object
     */
    public function get_parameters(string $request_param = null, bool $get_default = false)
    {
        // We don't return single values for dimension params
        if (array_key_exists($request_param, $this->dimension_params)) {
            return null;
        }

        $temp = new \stdClass;

        // Get dimension based tag parameters
        foreach ($this->dimension_params as $param => $value) {
            $temp->{$param} = ee()->TMPL->fetch_param($param, $this->dimension_params[$param]);
        }

        // Process a single parameter request
        if (is_string($request_param) && strlen($request_param)) {
            if ($get_default) {
                return $this->valid_params[$request_param];
            } else {
                $request_param_value = ee()->TMPL->fetch_param($request_param, $this->valid_params[$request_param]);
                if ($request_param_value != $this->valid_params[$request_param]) {
                    $request_param_value = $this->validate_parameters($request_param, $request_param_value);
                }
                return $request_param_value;
            }
        }

        // Get all valid parameters
        foreach ($this->valid_params as $param => $value) {
            // prefix variable with '@' to supress notices about missing indicies where these values
            // are not previously set within add-on
            @$temp->{$param} = ee()->TMPL->fetch_param($param, $this->valid_params[$param]);
            // if bg_color convert to valid colour format
            if ($param == 'bg_color') {
                $temp->{$param} = $this->validate_colour_string($temp->{$param});
            }
        }

        // Now do some validation / reconciliation and process any other tag parameters... 
        $consolidated_attributes = '';
        foreach (ee()->TMPL->tagparams as $param => $value) {
            if (array_key_exists($param, $this->dimension_params) || array_key_exists($param, $this->valid_params)) {
                $temp->{$param} = $this->validate_parameters($param, $value);
            } else {
                // put the non-valid attribute into the attributes parameter
                $consolidated_attributes .= ' ' . $param . '="' . $value . '"';
            }
        }

        // Now reconcile attributes and consolidated / pass-through attributes
        $temp->attributes .= $consolidated_attributes;

        // Check to see if auto_sharpen parameter set
        if (!is_null($temp->auto_sharpen) && strtolower(substr($temp->auto_sharpen, 0, 1)) == 'y') {
            // It is set, so adjust filter parameter if necessary... 
            if (is_null($temp->filter)) {
                // No filters defined yet so add auto_sharpen
                $temp->filter = 'auto_sharpen';
            } else {
                // Check to see if auto_sharpen already defined, if not append it... 
                if (!stripos($temp->filter, 'auto_sharpen')) {
                    $temp->filter .= '|auto_sharpen';
                }
            }
        }

        // Check to see if cache_dir parameter set and strip leading / if one present
        if (!is_null($temp->cache_dir)) {
            // This normalises all cache dirs to be relative to base_path directory
            $temp->cache_dir = ltrim($temp->cache_dir, '/');
        }

        // Check to see if disable_browser_checks parameter set
        if (!is_null($temp->disable_browser_checks) && strtolower(substr($temp->disable_browser_checks, 0, 1)) != 'y') {
            // It is not set to yes, so adjust filter parameter to be null regardless of what actually entered ... 
            $temp->disable_browser_checks = null;
        }

        // Check to see if consolidate classes / styles default set
        if (!is_null($temp->consolidate_class_style) && strtolower(substr($temp->consolidate_class_style, 0, 1)) != 'n') {
            // It is not set to no, so adjust parameter to be yes regardless of what actually entered ... 
            $temp->consolidate_class_style = 'y';
        }

        // $test = ee()->TMPL->fetch_param('src','not_set');
        return $temp;
    }

    /**
     * Utility function: Checks to see what image formats server can create
     * For now this assumes use of GD2 library... 
     *
     * @return	array	array with parameters
     */
    private function get_server_capabilities()
    {
        // Have we done this already on this instance?
        if (self::$valid_server_image_formats) {
            return self::$valid_server_image_formats;
        }

        // Get a list of formats supported by the GD library
        $server_gd_info = gd_info();

        // Work out what capabilities we have... 
        self::$valid_server_image_formats = [];
        foreach ($server_gd_info as $key => $value) {
            if (!in_array(strtolower(substr($key, 0, 2)), ['gd', 'fr', 'ji'])) {
                $this_capability = explode(' ', strtolower($key));
                if ($value === true && strtolower($this_capability[1]) != 'read') {
                    self::$valid_server_image_formats[] = $this_capability[0];
                    if ($this_capability[0] == 'jpeg') {
                        self::$valid_server_image_formats[] = 'jpg';
                    }
                }
            }
        }
        ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_server_capabilities'), self::$valid_server_image_formats);

        return self::$valid_server_image_formats;
    }

    /**
     * From https://stackoverflow.com/a/54827140/6475781
     * Estimates, if image has pixels with transparency. It shrinks image to 64 times smaller
     * size, if necessary, and searches for the first pixel with non-zero alpha byte.
     * If image has 1% opacity, it will be detected. If any block of 8x8 pixels has at least
     * one semi-opaque pixel, the block will trigger positive result. There are still cases,
     * where image with hardly noticeable transparency will be reported as non-transparent,
     * but it's almost always safe to fill such image with monotonic background.
     *
     * Icons with size <= 64x64 (or having square <= 4096 pixels) are fully scanned with
     * absolutely reliable result.
     *
     * @param  object $image // GDImage object or resource... 
     * @return bool
     */
    public function hasTransparency($image): bool
    {
        if (!is_resource($image) && !is_object($image)) {
            throw new \InvalidArgumentException("Image resource expected. Got: " . gettype($image));
        }

        $shrinkFactor      = 64.0;
        $minSquareToShrink = 64.0 * 64.0;

        $width  = imagesx($image);
        $height = imagesy($image);
        $square = $width * $height;

        if ($square <= $minSquareToShrink) {
            [$thumb, $thumbWidth, $thumbHeight] = [$image, $width, $height];
        } else {
            $thumbSquare = $square / $shrinkFactor;
            $thumbWidth  = (int) round($width / sqrt($shrinkFactor));
            $thumbWidth < 1 and $thumbWidth = 1;
            $thumbHeight = (int) round($thumbSquare / $thumbWidth);
            $thumb       = imagecreatetruecolor($thumbWidth, $thumbHeight);
            imagealphablending($thumb, false);
            imagecopyresized($thumb, $image, 0, 0, 0, 0, $thumbWidth, $thumbHeight, $width, $height);
        }

        for ($i = 0; $i < $thumbWidth; $i++) {
            for ($j = 0; $j < $thumbHeight; $j++) {
                if (imagecolorat($thumb, $i, $j) & 0x7F000000) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * PNG ALPHA CHANNEL SUPPORT for imagecopymerge(); 
     * This is a function like imagecopymerge but it handle alpha channel well!!! 
     * From http://www.php.net/manual/en/function.imagecopymerge.php
     *
     * @param  object $dst_im
     * @param  object $src_im
     * @param  int $dst_x
     * @param  int $dst_y
     * @param  int $src_x
     * @param  int $src_y
     * @param  int $src_w
     * @param  int $src_h
     * @param  float $pct
     * @return mixed
     */
    public static function imagecopymerge_alpha($dst_im, $src_im, $dst_x, $dst_y, $src_x, $src_y, $src_w, $src_h, $pct)
    {
        if (!isset($pct)) {
            return false;
        }
        $pct /= 100;
        // Get image width and height 
        $w = imagesx($src_im);
        $h = imagesy($src_im);
        // Turn alpha blending off 
        imagealphablending($src_im, false);
        // Find the most opaque pixel in the image (the one with the smallest alpha value) 
        $minalpha = 127;
        for ($x = 0; $x < $w; $x++)
            for ($y = 0; $y < $h; $y++) {
                $alpha = (imagecolorat($src_im, $x, $y) >> 24) & 0xFF;
                if ($alpha < $minalpha) {
                    $minalpha = $alpha;
                }
            }
        //loop through image pixels and modify alpha for each 
        for ($x = 0; $x < $w; $x++) {
            for ($y = 0; $y < $h; $y++) {
                //get current alpha value (represents the TRANSPARENCY!) 
                $colorxy = imagecolorat($src_im, $x, $y);
                $alpha = ($colorxy >> 24) & 0xFF;
                //calculate new alpha 
                if ($minalpha !== 127) {
                    $alpha = 127 + 127 * $pct * ($alpha - 127) / (127 - $minalpha);
                } else {
                    $alpha += 127 * $pct;
                }
                $alpha = (int) $alpha;
                //get the color index with new alpha 
                $alphacolorxy = imagecolorallocatealpha($src_im, ($colorxy >> 16) & 0xFF, ($colorxy >> 8) & 0xFF, $colorxy & 0xFF,  $alpha < 127 ? $alpha : 127);
                //set pixel with the new color + opacity 
                if (!imagesetpixel($src_im, $x, $y, $alphacolorxy)) {
                    return false;
                }
            }
        }
        // The image copy 
        imagecopy($dst_im, $src_im, (int) $dst_x, (int) $dst_y, (int) $src_x, (int) $src_y, (int) $src_w, (int) $src_h);
    }

    /**
     * Detects animated GIF from given file pointer resource or filename.
     * Derived from code at https://stackoverflow.com/a/42191495/6475781
     * This version works on file in memory rather than on disk.
     *
     * @param resource|string $file File pointer resource or filename
     * @return int|bool
     */
    function is_animated_gif($image)
    {
        if (!is_string($image)) {
            // Not an image string so bale out
            return false;
        }

        if (substr($image, 0, 3) !== "GIF") {
            // Not a GIF!
            return false;
        }

        // We use preg_match_all to count the markers for gif frames in the string
        // More than one and we've got an animated gif...
        // The test used is to look for the start of the 'application extension block' which
        // for animated gifs must always start with the bytes 21 F9 - if we get more than one 
        // we must have a layered GIF and so probably an animation.
        // Details here http://giflib.sourceforge.net/whatsinagif/animation_and_transparency.html

        $frames = preg_match_all('/\x21\xf9/', $image);

        return $frames > 1;
    }

    /**
     * Tests to see if pssed object is a GD resource or GDImage object
     *
     * @param  mixed  $var
     * @return bool
     */
    public function is_gd_image($var): bool
    {
        return ((gettype($var) == "object" && get_class($var) == "GdImage") || get_resource_type($var) == "gd");
    }

    /**
     * Utility function: Looks to see if image provided is in cache and still valid
     *
     * @param string $image_path
     * @return bool
     */
    public function is_image_in_cache(string $image_path)
    {
        // get the basename
        $filename = pathinfo($image_path)['basename'];

        // Default is that image is not in cache.
        $use_cache_copy = false;

        // Check to see if we have a cache copy and overwrite_cache option not set
        $cache_value = $this->get_parameters('cache');
        if (file_exists($image_path) && !$cache_value == 0 && strtolower(substr($this->get_parameters('overwrite_cache'), 0, 1)) == 'n') {
            $age_of_file = filemtime($image_path);
            $cache_duration_when_saved = $this->get_file_cache_duration($image_path);
            // check to see if cache timer has expired
            if (!$use_cache_copy =  $cache_duration_when_saved == -1 || ($cache_value > 0 && $cache_duration_when_saved && time() - $age_of_file < $cache_duration_when_saved)) {
                ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_not_found_in_cache'), $image_path);

                // If cache has expired delete the cache copy
                if (file_exists($image_path)) {
                    unlink($image_path);
                }
                // And delete entry from cache log
                $cache_log = ee('jcogs_img:Utilities')->cache_utility('get', JCOGS_IMG_CLASS . '/cache_log');
                if ($cache_log && isset($cache_log[$filename])) {
                    unset($cache_log[$filename]);
                }
            }
        } else {
            if ($cache_value == 0 || strtolower(substr($this->get_parameters('overwrite_cache'), 0, 1)) == 'y') {
                ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_cache_disabled'));
            } else {
                ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_not_found_in_cache'));
            }
        }

        if ($use_cache_copy) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_found_in_cache'), $filename);
            $this->update_cache_log($image_path);
        } else {
        }
        return $use_cache_copy;
    }

    /**
     * Checks to see if co-ordinates given are over an opaque pixel in an image
     *
     * @param  resource|\GdImage  $image
     * @param  int  $x
     * @param  int  $y
     * @return int|bool
     */
    public function is_pixel_opaque($image, int $x, int $y)
    {
        $x = round($x, 0);
        $y = round($y, 0);
        if (((gettype($image) == "object" && get_class($image) == "GdImage") || get_resource_type($image) == 'gd') && $x >= 0 && $x < imagesx($image) && $y >= 0 && $y < imagesy($image)) {
            return (imagecolorat($image, $x, $y) & 0x7F000000) >> 24 == 0;
        } else {
            return false;
        }
    }

    /**
     * Gets a listing of files within any directory called remote found within webroot or level below
     * or to remote directory path specified in settings
     *
     * @param  string $filename // Name of image we are looking for
     * @return string|bool // path to remote image or false
     */
    public function look_for_ce_image_remote_files(string $filename)
    {
        if (!is_string($filename)) {
            return false;
        }

        // Do we have a remote image cache directory?
        if (!ee('Filesystem')->exists(ee('jcogs_img:Utilities')->get_base_path() . ee('jcogs_img:Settings')::$settings['img_cp_ce_image_remote_dir'])) {
            return false;
        }

        // Do we have a copy of directory inventory in cache?
        $ce_image_remote_cache_listing = ee('jcogs_img:Utilities')->cache_utility('get', JCOGS_IMG_CLASS . '/ce_image_remote_cache_listing');

        if (!$ce_image_remote_cache_listing) {
            // Generate a new directory inventory
            $ce_image_remote_cache_listing = ee('Filesystem')->getDirectoryContents(ee('jcogs_img:Utilities')->get_base_path() . ee('jcogs_img:Settings')::$settings['img_cp_ce_image_remote_dir'], true);

            // if we got something, save it to the cache
            if ($ce_image_remote_cache_listing) {
                ee('jcogs_img:Utilities')->cache_utility('save', JCOGS_IMG_CLASS . '/ce_image_remote_cache_listing', $ce_image_remote_cache_listing, 60);
                // Save it to cache
            }
        }

        if ($ce_image_remote_cache_listing) {
            // Search directory inventory for image
            $results = array_filter($ce_image_remote_cache_listing, function ($value, $key) use ($filename) {
                // we want files that match the filename and are images
                return stripos($value, $filename) && exif_imagetype($value);
            }, ARRAY_FILTER_USE_BOTH);
            if ($results) {
                $path = '';
                $max_size = 0;
                foreach ($results as $result) {
                    $size = getimagesize($result);
                    if ($size) {
                        $max_size = max($max_size, $size[0] + $size[1]);
                        if ($max_size == $size[0] + $size[1]) {
                            $path = $result;
                        }
                    }
                }
                // Get path of image found and return
                return $path;
            }
        }
        return false;
    }

    /**
     * Wraps a string to a given number of pixels.
     * From here: https://www.php.net/manual/en/function.wordwrap.php#116467
     * Code modified slightly to update to php7.4 compatibility
     *
     * This function operates in a similar fashion as PHP's native wordwrap function; however,
     * it calculates wrapping based on font and point-size, rather than character count. This
     * can generate more even wrapping for sentences with a consider number of thin characters.
     *
     * @static $mult;
     * @param string $text - Input string.
     * @param int $width - Width, in pixels, of the text's wrapping area.
     * @param float $size - Size of the font, expressed in pixels.
     * @param string $font - Path to the typeface to measure the text with.
     * @param float $line_height - line height to use.
     * @return array $return[0] - The original string with wrapping.
     *               $return[1] - The estimated height of the text block based on rows*line_height
     */
    public function pixel_word_wrap(string $text, int $width, float $size, string $font, float $line_height)
    {

        # Passed a blank value? Bail early.
        if (!$text || strlen($text) == 0) return ["", 0];

        # Check if imagettfbbox is expecting font-size to be declared in points or pixels.
        static $mult = 0.8;
        // if (!$mult) {
        //     $mult = version_compare(GD_VERSION, '2.0', '>=') ? .75 : 1;
        // }

        # See if text already fits the designated space without wrapping.
        $box = imageftbbox($size * $mult, 0, $font, $text);
        if (($box[2] - $box[0]) / $mult < $width) return [$text, 1];

        # Start measuring each line of our input and inject line-breaks when overflow's detected.
        $output = '';
        $breakpoint = 0;
        $length = 0;
        $row_count = 1;

        $words = preg_split('/\b(?=\S)|(?=\s)/', $text);
        $word_count = count($words);
        for ($i = 0; $i < $word_count; ++$i) {

            # Newline
            if (PHP_EOL === $words[$i]) {
                $row_count++;
                $length = 0;
            }

            # Strip any leading tabs.
            if (!$length) {
                $words[$i] = preg_replace('/^\t+/', '', $words[$i]);
            }

            $box = imageftbbox($size * $mult, 0, $font, $words[$i], ['linespacing' => $line_height]);
            $m = ($box[2] - $box[0]);

            # This is one honkin' long word, so try to hyphenate it.
            if (($diff = $width - $m) <= 0) {
                $diff = abs($diff);

                # Figure out which end of the word to start measuring from. Saves a few extra cycles in an already heavy-duty function.
                if ($diff - $width <= 0) for ($s = strlen($words[$i]); $s; --$s) {
                    $box = imageftbbox($size * $mult, 0, $font, substr($words[$i], 0, $s) . '-');
                    if ($width > (($box[2] - $box[0])) + $size) {
                        $breakpoint = $s;
                        break;
                    }
                }
                else {
                    $word_length = strlen($words[$i]);
                    for ($s = 0; $s < $word_length; ++$s) {
                        $box = imageftbbox($size * $mult, 0, $font, substr($words[$i], 0, $s + 1) . '-');
                        if ($width < (($box[2] - $box[0])) + $size) {
                            $breakpoint = $s;
                            break;
                        }
                    }
                }

                if ($breakpoint) {
                    $w_l = substr($words[$i], 0, $s + 1) . '-';
                    $w_r = substr($words[$i], $s + 1);

                    $words[$i] = $w_l;
                    array_splice($words, $i + 1, 0, $w_r);
                    ++$word_count;
                    $box = imageftbbox($size * $mult, 0, $font, $w_l);
                    $m = ($box[2] - $box[0]);
                }
            }
            # If there's no more room on the current line to fit the next word, start a new line.
            if ($length > 0 && $length + $m >= $width) {
                $output .= PHP_EOL;
                $row_count++;
                $length = 0;

                # If the current word is just a space, don't bother. Skip (saves a weird-looking gap in the text).
                if (' ' === $words[$i]) continue;
            }
            # Write another word and increase the total length of the current line.
            $output .= $words[$i];
            $length += $m;
        }
        # Get dimensions of text box
        $box = imageftbbox($size * $mult, 0, $font, $output, ['linespacing' => $line_height]);
        return [$output, $row_count];
    }

    /**
     * Sharpen image
     *
     * @param  object $image // Imagine Image object
     * @param float $sharpening_value
     * @return object
     */
    public function sharpen($image, float $sharpening_value = 10)
    {
        $amount = max(0, min(100, $sharpening_value));

        // build matrix
        $min = $amount >= 10 ? $amount * -0.01 : 0;
        $max = $amount * -0.025;
        $abs = ((4 * $min + 4 * $max) * -1) + 1;
        $div = 1;

        $matrix = [
            [$min, $max, $min],
            [$max, $abs, $max],
            [$min, $max, $min]
        ];

        // apply the matrix
        $gdimage = $image->getGdResource();
        if (imageconvolution($gdimage, $matrix, $div, 0)) {
            $newimage = $this->convert_GDImage_object_to_image($gdimage);
            return $newimage;
        }
        ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_sharpen_failed'));
        return $image;
    }

    /**
     * Utility function: Update image cache log for nominated image.
     *
     * @param string $image_path
     * @param float $processing_time
     * @param string|null $cache_dir
     * @return void
     */
    public function update_cache_log(string $image_path, float $processing_time = 0, string $cache_dir = null, string $source_path = null)
    {
        // get the basename
        $filename = pathinfo($image_path)['basename'];

        // Get the tracker file for this cache 
        $cache_log = ee('jcogs_img:Utilities')->cache_utility('get', JCOGS_IMG_CLASS . '/cache_log');
        // Do we need to create a cache log?
        if (!isset($cache_log[$filename]) || $cache_log === false) {
            // No cache log, or no entry for this key so create one
            if ($cache_log === false) {
                $cache_log = [];
                $cache_log['inception_date'] = time();
            }
            $cache_log[$filename]['count'] = 0;
            $cache_log[$filename]['size'] = file_exists($image_path) ? @filesize($image_path) : null;
            $cache_log[$filename]['processing_time'] = $processing_time;
            $cache_log[$filename]['cummulative_size'] = 0;
            $cache_log[$filename]['cummulative_processing_time'] = 0;
            $cache_log[$filename]['cache_dir'] = $cache_dir ?: $this->settings['img_cp_default_cache_directory'];
            $cache_log[$filename]['sourcepath'] = $source_path ?: '';
        } else {
            // Cache log file exists, so just update entry
            $cache_log[$filename]['count']++;
            $cache_log[$filename]['cummulative_size'] += $cache_log[$filename]['size'];
            $cache_log[$filename]['cummulative_processing_time'] += $cache_log[$filename]['processing_time'];
            if (! array_key_exists('sourcepath',$cache_log)) {
                $cache_log[$filename]['sourcepath'] = $source_path ?: '';
            }
        }
        // Update the cache log (save for one year)
        ee('jcogs_img:Utilities')->cache_utility('save', JCOGS_IMG_CLASS . '/cache_log', $cache_log, 31536000);
    }

    /**
     * Utility function: Normalises colour strings / rgba forms to Imagine RGB colour pallette 
     * If colour not valid or not set returns default background colour
     * Uses some code inspired by that found here https://mekshq.com/how-to-convert-hexadecimal-color-code-to-rgb-or-rgba-using-php
     *
     * @param string $colour
     * @param float $opacity
     * @return object|string
     */
    public function validate_colour_string(string $colour = null, float $opacity = 1)
    {

        // Check to see if we have been here before... 
        if (!is_string($colour)) {
            return $colour;
        }

        $default = isset(ee('jcogs_img:ImageUtilities')::$current_params['bg_color']) && ee('jcogs_img:ImageUtilities')::$current_params['bg_color'] != '' ? ee('jcogs_img:ImageUtilities')::$current_params['bg_color'] : ee('jcogs_img:Settings')::$settings['img_cp_default_bg_color'];

        //Use default if no color provided
        $colour = $colour ?? $default;

        //Sanitize $colour if "#" is provided 
        // Need to check for hex and rgb and rgba forms.
        if (stripos($colour, '#') !== false) {
            $colour = substr($colour, 1);
        }

        // Check to see if colour is in rgb(a) format
        // If opacity given use that rather than any opacity given in call
        if (strtolower(substr($colour, 0, 4)) == 'rgba') {
            if (preg_match('/rgba\((.*)\)/', $colour, $matches) == 0) {
                ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_invalid_colour_string'), $colour);
                return $this->validate_colour_string($default);
            };
            $rgb = explode(',', $matches[1]);
            if (count($rgb) == 4) {
                $opacity = trim(array_pop($rgb));
            } else {
                $opacity = $opacity ?? 1;
            }
        } elseif (strtolower(substr($colour, 0, 3)) == 'rgb') {
            // Check to see if colour is in rgb format
            if (preg_match('/rgb\((.*)\)/', $colour, $matches) == 0) {
                ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_invalid_colour_string'), $colour);
                return $this->validate_colour_string($default);
            };
            $rgb = explode(',', $matches[1]);
        } elseif (strlen($colour) == 8) {
            //Check if colour is in hexadecimal format and has 8, 6, 4 or 3 characters and get values
            //If colour has 8 or 4 then use that in preference to any opacity value given in call
            $rgb = array_map('hexdec', array($colour[0] . $colour[1], $colour[2] . $colour[3], $colour[4] . $colour[5]));
            $opacity = round(hexdec($colour[6] . $colour[7]) / 255, 2);
        } elseif (strlen($colour) == 6) {
            $rgb = array_map('hexdec', array($colour[0] . $colour[1], $colour[2] . $colour[3], $colour[4] . $colour[5]));
        } elseif (strlen($colour) == 4) {
            $rgb = array_map('hexdec', array($colour[0] . $colour[0], $colour[1] . $colour[1], $colour[2] . $colour[2]));
            $opacity = round(hexdec($colour[4] . $colour[4]) / 255, 2);
        } elseif (strlen($colour) == 3) {
            $rgb = array_map('hexdec', array($colour[0] . $colour[0], $colour[1] . $colour[1], $colour[2] . $colour[2]));
        } else {
            return $this->validate_colour_string($default);
        }

        // Normalise rgb values if necessary
        $rgb[0] = intval($rgb[0]);
        $rgb[1] = intval($rgb[1]);
        $rgb[2] = intval($rgb[2]);
        $rgb[0] = $rgb[0] > 255 || !is_int($rgb[0]) ? '255' : $rgb[0];
        $rgb[1] = $rgb[1] > 255 || !is_int($rgb[1]) ? '255' : $rgb[1];
        $rgb[2] = $rgb[2] > 255 || !is_int($rgb[2]) ? '255' : $rgb[2];
        $rgb[0] = $rgb[0] < 0 ? '0' : $rgb[0];
        $rgb[1] = $rgb[1] < 0 ? '0' : $rgb[1];
        $rgb[2] = $rgb[2] < 0 ? '0' : $rgb[2];

        //Normalise opacity value if needed and return value in rgb(a) format
        $opacity = $opacity > 1 ? 1 : $opacity;
        $opacity = $opacity < 0 ? 0 : $opacity;

        // Now scale opacity to 0-100 range to suit Imagine library
        $opacity = (int) round($opacity * 100, 0);
        return (new Palette\RGB())->color([$rgb[0], $rgb[1], $rgb[2]], $opacity);
    }

    /**
     * Utility function: Adjusts the value of a parameter to pixel value without units, by either:
     *  * removing px from end of string
     *  * converting % value to pixel value.
     *
     * @param string $param
     * @param int $base_length
     * @return int|bool|null
     */
    public function validate_dimension(string $param = null, $base_length = null)
    {
        // If we get null return null
        if (is_null($param) || (is_string($param) && strlen($param) == 0)) {
            return null;
        }

        // Ensure we have an integer value
        $base_length = $base_length ? round($base_length,0) : $base_length;

        // Is it at a %?
        if (substr($param, -1) == '%') {
            // If we get a % and no base_length return empty handed
            return $base_length ? round($base_length * intval($param) / 100,0) : false;
        }
        
        // Is a px?
        if (substr($param, -2) == 'px') {
            return intval($param);
        }
        // See if it is a string or integer == 0
        if ($param == '0' || $param == 0) {
            return 0;
        }
        // Cast to integer - if not an integer it will give 0
        if ((int) $param != 0) {
            return (int) $param;
        }
        // Not sure what it is so bale out!
        ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_invalid_dimension'), $param);
        return false; // Oops!
    }

    /**
     * Utility function: that an image format will be accepted by current browser
     * 
     * If image format is not accepted, returns false instead.
     * 
     * @param string $image_format
     * @return bool $image_format
     */
    public function validate_browser_image_format(string $image_format)
    {
        // Check to see if image format is one the browser can read
        // If valid, or checking is disabled, return true
        // If not valid and checking enabled, return false
        $valid_format = $this->settings['img_cp_enable_browser_check'] == 'y' ? in_array($image_format, $this->get_browser_image_capabilities()) : true;
        return $valid_format;
    }

    /**
     * Utility function: validate some parameters that have complex needs
     * 
     * @param array $parameters
     * @return mixed $image_format
     */
    private function validate_parameters(string $param, $value)
    {
        // its a valid parameter
        // is it a valid src?
        if ($param == 'src' || $param == 'fallback_src') {
            $filename = ee()->TMPL->fetch_param($param, $this->valid_params[$param]);
            // Just in case src parameter value coming from stash, urldecode it ... 
            $is_encoded = preg_match('~%[0-9A-F]{2}~i', $filename);
            // Is encoded is true if filename contains typical urlencoded artefacts (e.g. %2F)
            if ($is_encoded) {
                // Encoded, so swap + and = for encoded equivalents and then decode
                $value = urldecode(str_replace(['+', '='], ['%2B', '%3D'], $filename));
            } else {
                $value = $filename;
            }
            // Check to see if it is a text version of an EE file field
            if ($ee_filedir_test = ee('jcogs_img:Utilities')->parseFiledir($value)) {
                $value = $ee_filedir_test != '' ? $ee_filedir_test : $value;
            }
        }
        // is it a valid colour?
        if ($param == 'bg_color') {
            $value = $this->validate_colour_string($value);
        }
        // is it a valid filename or extension / suffix?
        if (in_array($param, ['filename', 'filename_prefix', 'filename_suffix'])) {
            // We cannot allow these to contain any URI incompatible characters
            // This can get very complicated
            // So we dodge the issue and use EE's built in tool...
            ee()->load->library('api');
            $value = ee()->legacy_api->make_url_safe($value);
        }
        // is it a valid cache_dir path?
        if ($param == 'cache_dir') {
            $value = trim($value, '/');
            // We don't allow the cache_dir to be the root, so check that first
            $value = $value != '' ? $value : $this->valid_params['cache_dir'];
            if (!ee('Filesystem')->exists(ee('jcogs_img:Utilities')->path($value, true))) {
                // Cache path provided does not exist and cannot be created
                // So put a note in debug log and set path back to default
                ee('jcogs_img:Utilities')->debug_message(sprintf(lang('jcogs_img_cache_path_not_found'), $value));
                $value = $this->valid_params[$param];
            }
        }
        // is it a valid face_detect sensitivity?
        if ($param == 'face_detect_sensitivity') {
            $value = max(min(intval($value),9),1);
        }
        return $value;
    }


    /**
     * Utility function: that an image format will be accepted by current server
     * 
     * If image format is not accepted, returns false.
     * 
     * @param string $image_format
     * @return bool 
     */
    public function validate_server_image_format(string $image_format)
    {
        // Check to see if image format is one the server can work with
        // if not, return false
        return in_array($image_format, $this->get_server_capabilities());
    }


    /**
     * Utility function: convert an rgb value to an HSL format
     * from https://gist.github.com/brandonheyer/5254516
     * 
     * @param int $r // R value in RGB
     * @param int $g // G value in RGB
     * @param int $b // B value in RGB
     * @return array
     */
    public function rgbToHsl(int $r, int $g, int $b)
    {

        $oldR = min(max($r, 0), 255);
        $oldG = min(max($g, 0), 255);
        $oldB = min(max($b, 0), 255);

        $r /= 255;
        $g /= 255;
        $b /= 255;

        $max = max($r, $g, $b);
        $min = min($r, $g, $b);

        $h = 0;
        $s = 0;
        $l = ($max + $min) / 2;
        $d = $max - $min;

        if ($d == 0) {
            $h = $s = 0; // achromatic
        } else {
            $s = $d / (1 - abs(2 * $l - 1));

            switch ($max) {
                case $r:
                    $h = 60 * fmod((($g - $b) / $d), 6);
                    if ($b > $g) {
                        $h += 360;
                    }
                    break;

                case $g:
                    $h = 60 * (($b - $r) / $d + 2);
                    break;

                case $b:
                    $h = 60 * (($r - $g) / $d + 4);
                    break;
            }
        }

        return array(round($h, 3), round($s, 3), round($l, 3));
    }

    /**
     * Utility function: convert an hsl value to an RGB format
     * from https://gist.github.com/brandonheyer/5254516
     * 
     * @param float $h // H value in HSL
     * @param float $s // S value in HSL
     * @param float $l // L value in HSL
     * @return array
     */
    function hslToRgb(float $h, float $s, float $l)
    {

        $h = min(max($h, 0), 360);
        $s = min(max($s, 0), 1);
        $l = min(max($l, 0), 1);

        $r = 0;
        $g = 0;
        $b = 0;

        $c = (1 - abs(2 * $l - 1)) * $s;
        $x = $c * (1 - abs(fmod(($h / 60), 2) - 1));
        $m = $l - ($c / 2);

        if ($h < 60) {
            $r = $c;
            $g = $x;
            $b = 0;
        } else if ($h < 120) {
            $r = $x;
            $g = $c;
            $b = 0;
        } else if ($h < 180) {
            $r = 0;
            $g = $c;
            $b = $x;
        } else if ($h < 240) {
            $r = 0;
            $g = $x;
            $b = $c;
        } else if ($h < 300) {
            $r = $x;
            $g = 0;
            $b = $c;
        } else {
            $r = $c;
            $g = 0;
            $b = $x;
        }

        $r = ($r + $m) * 255;
        $g = ($g + $m) * 255;
        $b = ($b + $m) * 255;

        return array(floor($r), floor($g), floor($b));
    }
}
