<?php

/**
 * Settings Service
 * ================
 * Service to set the settings for the JCOGS Image add-on
 * =====================================================
 *
 * @category   ExpressionEngine Add-on
 * @package    JCOGS Image
 * @author     JCOGS Design <contact@jcogs.net>
 * @copyright  Copyright (c) 2021 - 2023 JCOGS Design
 * @license    https://jcogs.net/add-ons/jcogs_img/license.html
 * @version    1.3.21.1
 * @link       https://JCOGS.net/
 * @since      File available since Release 1.0.0
 */

/**
 * Generic path to multiple libraries code (for Imagine reference)
 * 
 * Also here - https://stackoverflow.com/questions/534159/instantiate-a-class-from-a-variable-in-php
 * 
 * $string = 'Imagine\\'.$library.'\\Imagine'
 * "Imagine\\Imagine"
 * $library = 'Gd'
 * $string = 'Imagine\\'.$library.'\\Imagine'
 * $test = new $string()
 */


namespace JCOGSDesign\Jcogs_img\Service;

class Settings
{
    public static $settings;

    public function __construct()
    {
        if (empty(self::$settings)) {
            self::$settings = $this->get_settings();
        }
    }

    private function _default_settings()
    {
        return array(
            'jcogs_add_on_class'                          => JCOGS_IMG_CLASS,
            'jcogs_add_on_name'                           => JCOGS_IMG_NAME,
            'jcogs_add_on_version'                        => JCOGS_IMG_VERSION,
            'enable_img'                                  => 'y',
            'jcogs_license_key'                           => '',
            'jcogs_license_key_email'                     => '',
            'jcogs_staging_domain'                        => '',
            'jcogs_license_mode'                          => 'invalid',
            'jcogs_license_server_domain'                 => 'mule.jcogs.net',
            'img_cp_default_cache_directory'              => 'images/jcogs_img/cache',
            'img_cp_path_prefix'                          => '',
            'img_cp_default_cache_duration'               => '2678400',
            'img_cp_default_cache_audit_after'            => '604800',
            'img_cp_enable_debugging'                     => 'n',
            'img_cp_enable_browser_check'                 => 'y',
            'img_cp_default_filename_separator'           => '_-_',
            'img_cp_enable_cache_audit'                   => 'y',
            'img_cp_cache_auto_manage'                    => 'n',
            'img_cp_default_max_source_filename_length'   => '200',
            'img_cp_ce_image_remote_dir'                  => 'images/remote',
            'img_cp_default_max_image_size'               => '4',
            'img_cp_default_min_php_ram'                  => '64',
            'img_cp_default_min_php_process_time'         => '60',
            'img_cp_default_php_remote_connect_time'      => '3',
            'img_cp_default_user_agent_string'            => 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10.15; rv:98.0) Gecko/20100101 Firefox/98.0',
            'img_cp_default_image_format'                 => 'source',
            'img_cp_jpg_default_quality'                  => '90',
            'img_cp_png_default_quality'                  => '6',
            'img_cp_default_bg_color'                     => '#FFFFFF',
            'img_cp_enable_svg_passthrough'               => 'n',
            'img_cp_default_img_width'                    => '350',
            'img_cp_default_img_height'                   => '150',
            'img_cp_allow_scale_larger_default'           => 'n',
            'img_cp_class_consolidation_default'          => 'y',
            'img_cp_attribute_variable_expansion_default' => 'y',
            'img_cp_class_always_output_full_urls'        => 'n',
            'img_cp_enable_auto_sharpen'                  => 'n',
            'img_cp_enable_lazy_loading'                  => 'n',
            'img_cp_lazy_loading_mode'                    => 'lqip',
            'img_cp_lazy_progressive_enhancement'         => '1',
            'img_cp_enable_default_fallback_image'        => 'n',
            'img_cp_fallback_image_colour'                => '#306392',
            'img_cp_fallback_image_local'                 => '',
            'img_cp_fallback_image_remote'                => 'https://placekitten.com/550/350',
            'img_cp_enable_auto_adjust'                   => 'n',
            'img_cp_default_max_image_dimension'          => '0',
            'img_cp_ignore_save_type_for_animated_gifs'   => '0'
        );
    }

    /**
     * Returns the app settings as an array, filling in default values where there are none saved
     *
     * @return array
     */
    private function get_settings()
    {
        if (self::$settings) {
            return self::$settings;
        }

        $query = ee()->db->get_where('jcogs_img_settings', array('site_id' => ee()->config->item('site_id')));
        self::$settings = array();
        if ($query->num_rows() > 0) {
            foreach ($query->result_array() as $row) {
                self::$settings[$row["setting_name"]] = $row["value"];
            }
            self::$settings = array_merge($this->_default_settings(), self::$settings);
        } else {
            self::$settings = $this->_default_settings();
        }
        return self::$settings;
    }

    /**
     * Saves app settings
     *
     * @param array $settings
     * @return bool
     */
    public function save_settings($settings = array())
    {
        // New settings are the merger of current settings and inbound settings
        $new_settings = array_merge(self::$settings, $settings);

        // Clear  the licensing server action table in case we've changed domain
        ee('jcogs_img:Utilities')->cache_utility('delete', JCOGS_IMG_CLASS . '/' . 'licensing_action_ids');

        // Get what is in data table - set to array if nothing there
        $data_in_table = [];
        $query = ee()->db->get_where('jcogs_img_settings', array('site_id' => ee()->config->item('site_id')));
        if ($query->num_rows) {
            $query_results = $query->result_array();
            foreach ($query_results as $row) {
                $data_in_table[$row['setting_name']] = $row['value'];
            }
        }

        // Loop through the new settings and see if we have same thing saved.
        foreach ($new_settings as $key => $value) {
            // Work out what is in data table and update to new values as required
            if (!isset($data_in_table[$key])) {
                // Value is not in table so add it to the table
                ee()->db->insert('jcogs_img_settings', array('site_id' => ee()->config->item('site_id'), 'setting_name' => $key, 'value' => $value));
            } elseif ($data_in_table[$key] != $value) {
                // There is something in datatable, but update it as new value is different
                ee()->db->update('jcogs_img_settings', array('value' => $value), array('site_id' => ee()->config->item('site_id'), 'setting_name' => $key));
            }
        }
        // Update static value with new settings
        self::$settings = $new_settings;

        return true;
    }
}
