<?php

/**
 * JCOGS Utilities Service
 * =======================
 * Service for common services used by JCOGS add-ons
 * =====================================================
 *
 * @category   ExpressionEngine Add-on
 * @package    JCOGS Utilities
 * @author     JCOGS Design <contact@jcogs.net>
 * @copyright  Copyright (c) 2021 - 2023 JCOGS Design
 * @license    https://jcogs.net/add-ons/jcogs_img/license.html
 * @version    1.0.0
 * @link       https://JCOGS.net/
 * @since      File available since Release 1.0.0
 */

namespace JCOGSDesign\Jcogs_img\Service;

ee()->lang->load('jcogs_utils', ee()->session->get_language(), false, true, PATH_THIRD . 'jcogs_img/');

class Utilities
{
    public $settings;
    public static $browser_image_format_support;
    public static $valid_server_image_formats;
    public static $license_status;
    private $_cache_path;

    public function __construct()
    {
        $this->settings = ee('jcogs_img:Settings')::$settings;
        ee()->load->helper('file');
        $this->_cache_path = PATH_CACHE;
    }

    /**
     * Checks that allow_url_fopen is enabled
     *
     * @return boolean
     */
    public function allow_url_fopen_enabled()
    {
        return ini_get('allow_url_fopen');
    }

    /**
     * JCOGS Utility - work-around for EE cache set to dummy 
     * =====================================================
     * Convert text to UTF-8 coding (to ensure works OK with translator services)
     * Or throws an error
     *
     * @param string $operation
     * @param string $key
     * @param mixed $value
     * @param int $ttl
     * @param string $scope
     * @return string
     */
    public function cache_utility(string $operation = '', string $key = null, $data = null, int $ttl = 60, $scope = \Cache::LOCAL_SCOPE)
    {
        // Check we have a location
        if (!$key) {
            $this->debug_message(lang('jcogs_utils_no_cache_key_provided'));
            return false;
        }

        // Execute requested operation
        switch ($operation) {
            case 'get':
                $key = $this->_namespaced_key($key, $scope);
                if (!file_exists($this->_cache_path . $key)) {
                    return false;
                }
                $data = @unserialize(file_get_contents($this->_cache_path . $key));
                if (!is_array($data)) {
                    return false;
                }
                if ($data['ttl'] > 0 && ee()->localize->now > $data['time'] + $data['ttl']) {
                    if (file_exists(($this->_cache_path . $key))) unlink($this->_cache_path . $key);
                    return false;
                }
                return $data['data'];

                case 'save':
                $contents = array(
                    'time' => ee()->localize->now,
                    'ttl' => $ttl,
                    'data' => $data
                );
                $key = $this->_namespaced_key($key, $scope);
                // Build file path to this key
                $path = $this->_cache_path . $key;
                // Remove the cache item name to get the path by looking backwards
                // for the directory separator
                $path = substr($path, 0, strrpos($path, DIRECTORY_SEPARATOR) + 1);
                // Create namespace directory if it doesn't exist
                if (!file_exists($path) or !is_dir($path)) {
                    @mkdir($path, 0777, true);
                    // Grab the error if there was one
                    $error = error_get_last();
                    // If we had trouble creating the directory, it's likely due to a
                    // concurrent process already having created it, so we'll check
                    // to see if that's the case and if not, something else went wrong
                    // and we'll show an error
                    if (!is_dir($path) or !is_writable($path)) {
                        trigger_error($error['message'], E_USER_WARNING);
                    } else {
                        // Write an index.html file to ensure no directory indexing
                        write_index_html($path);
                    }
                }
                if (write_file($this->_cache_path . $key, serialize($contents))) {
                    @chmod($this->_cache_path . $key, 0666);
                    return true;
                }
                return false;

                case 'delete':
                $path = $this->_cache_path . $this->_namespaced_key($key, $scope);

                // If we are deleting contents of a namespace
                if (strrpos($key, \Cache::NAMESPACE_SEPARATOR, strlen($key) - 1) !== false) {
                    $path .= DIRECTORY_SEPARATOR;
                    if (delete_files($path, true)) {
                        // Try to remove the namespace directory; it may not be
                        // removeable on some high traffic sites where the cache fills
                        // back up quickly
                        @rmdir($path);
                        return true;
                    }
                    return false;
                }
                return file_exists($path) ? unlink($path) : false;

                default:
                $this->debug_message(lang('jcogs_utils_unknown_cache_operation_requested'));
                return false;
        }
    }

    /**
     * JCOGS Utility - convert to utf-8 
     * ==========================================
     * Convert text to UTF-8 coding (to ensure works OK with translator services)
     * Or throws an error
     *
     * @param  string $content_to_convert
     * @return string
     */
    public function convert_to_utf_8(string $content_to_convert)
    {
        if (
            !mb_check_encoding($content_to_convert, 'UTF-8')
            or !($content_to_convert === mb_convert_encoding(mb_convert_encoding($content_to_convert, 'UTF-32', 'UTF-8'), 'UTF-8', 'UTF-32'))
        )
        // If it fails this test then content is not in UTF-8 format so we need to convert...
        {
            $content_to_convert = mb_convert_encoding($content_to_convert, 'UTF-8');

            // Check that conversion was successful
            if (mb_check_encoding($content_to_convert, 'UTF-8')) {
                $this->debug_message(lang('jcogs_utils_UTF8_converted_text')); // Whoop! It worked.
            } else {
                return ee()->output->fatal_error(__METHOD__ . ' ' . lang('UTF8_conversion_failed')); // Oops!
            }
        }
        return $content_to_convert;
    }

    /**
     * Utility to return elapsed time between date value provided and now
     *
     * @param  integer $date
     * @return string
     */
    public static function date_difference_to_now(int $date)
    {
        if (!is_int($date) || $date > time() || $date < 0) {
            return false;
        }
        $diff = time() - $date;

        switch ($diff) {
            case $diff < 60:
                $value = $diff;
                $units = $value === 1 ? ' second' : ' seconds';
                break;
            case $diff < 3600:
                $value = round($diff / 60);
                $units = $value < 2 ? ' minute' : ' minutes';
                break;
            case $diff < 86400:
                $value = round($diff / 3600);
                $units = $value < 2 ? ' hour' : ' hours';
                break;
            case $diff < 604800:
                $value = round($diff / 86400);
                $units = $value < 2 ? ' day' : ' days';
                break;
            case $diff < 2600640:
                $value = round($diff / 604800);
                $units = $value < 2 ? ' week' : ' weeks';
                break;
            case $diff < 31207680:
                $value = round($diff / 2600640);
                $units = $value < 2 ? ' month' : ' months';
                break;
            default:
                $value = round($diff / 31207680);
                $units = $value < 2 ? ' year' : ' years';
        }
        return $value . $units;
    }

    /* JCOGS Image - debug message utility 
     * ===================================
     * If debug is enabled, write messages to the debug log
     * Params:
     * - $msg       The text to write to debug log
     */
    public static function debug_message(string $msg, $details = null)
    {
        if (ee('jcogs_img:Settings')::$settings['img_cp_enable_debugging'] === 'y' && REQ == 'PAGE' && isset(ee()->TMPL)) {
            $dbt = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS, 2);
            $caller = lang('jcogs_img_module_name');
            if (is_array($details)) {
                ee()->TMPL->log_item($caller . ' (' . $dbt[1]['function'] . ') ' . '<span style=\'color:darkblue\'>' . $msg . '</span>', $details);
            } else {
                ee()->TMPL->log_item($caller . ' (' . $dbt[1]['function'] . ') ' . '<span style=\'color:darkblue\'>' . $msg . '</span> <span style=\'color:var(--ee-link)\'>' . $details . '</span>');
            }
        }
    }

    /**
     * Look up first value that is less than the lookup value supplied
     *
     * @param  float $lookupValue
     * @param  array $array
     * @return mixed
     */
    public function vlookup($lookupValue, $array)
    {

        $result = false;
        $keys = array_keys($array);

        //test each set against the $lookupValue variable,
        //and set/reset the $result value

        for ($i = 0; $i < count($array) - 1; $i++) {
            if ($lookupValue > $keys[$i] && $lookupValue <= $keys[$i + 1]) {
                $result = $array[$keys[$i]];
                break;
            }
        }

        return $result;
    }

    /**
     * Utility function: lookup values in array
     * based on code from 
     * https://stackoverflow.com/a/22750841/6475781
     *
     * @param  array  $array
     * @param  float  $value
     * @param  boolean $exact
     * @return array|bool
     */
    public function fast_nearest($array, $value, $exact = false)
    {
        if (isset($array[round($value, 0)])) {
            // If exact match found, and searching for exact (not nearest), return result.
            return array($value => $array[$value], 'exact' => true);
        } elseif ($exact || empty($array)) {
            return false;
        }
        // else
        $keys = array_keys($array);
        $min = $keys[0];
        $s = 0;
        $max = end($keys);
        $e = key($keys);
        if ($s == $e) {
            // only one element, it's closest
            return array_merge($array, array('exact' => false));
        } elseif ($value < $min) {
            return array($min => $array[$min], 'exact' => false);
        } elseif ($value > $max) {
            return array($max => $array[$max], 'exact' => false);
        }
        $result = false;
        do {
            $guess = $s + (int)(($value - $min) / ($max - $min) * ($e - $s));
            if ($guess < $s) {
                // oops, off the scale; we found it
                $result = $keys[$s];
            } elseif ($guess > $e) {
                $result = $keys[$e];
            } elseif ($keys[$guess] > $value && $keys[$guess - 1] < $value) {
                // found range
                $result = (($value - $keys[$guess - 1]) < ($keys[$guess] - $value)
                    ? $keys[$guess - 1]
                    : $keys[$guess]);
            } elseif ($keys[$guess] < $value && $keys[$guess + 1] > $value) {
                $result = (($value - $keys[$guess]) < ($keys[$guess + 1] - $value)
                    ? $keys[$guess]
                    : $keys[$guess + 1]);
            } elseif ($keys[$guess] > $value) {
                // narrowing search area
                $e = $guess - 1;
                $max = $keys[$e];
            } elseif ($keys[$guess] < $value) {
                $s = $guess + 1;
                $min = $keys[$s];
            }
        } while ($e != $s && $result === false);
        if ($result === false) {
            // throw new Exception("Math laws don't work in this universe.");
            return false;
        }
        return $array[$result];
    }

    /**
     * Utility function: Fixes units of bytes to proper values
     * based on code from 
     * https://stackoverflow.com/questions/2510434/format-bytes-to-kilobytes-megabytes-gigabytes
     *
     * @param integer $bytes
     * @param integer $precision
     * @return string
     */
    public function formatBytes($bytes, $precision = 2)
    {
        $units = array('B', 'KiB', 'MiB', 'GiB', 'TiB');

        $bytes = max(intval($bytes), 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);

        // Uncomment one of the following alternatives
        $bytes /= pow(1024, $pow);
        // $bytes /= (1 << (10 * $pow)); 

        return round($bytes, $precision) . ' ' . $units[$pow];
    }


    /**
     * Utility function: Get the basepath
     *
     * @return	string	$basepath
     */
    public function get_base_path()
    {
        // Is base_path blank or invalid?
        if (!ee()->config->item('base_path') || !file_exists(rtrim(ee()->config->item('base_path'), '/') . '/')) {
            // $basepath is invalid, so put a note into template debugger...
            $this->debug_message(lang('jcogs_utils_no_base_path'), rtrim(ee()->config->item('base_path'), '/') . '/');
            return false;
        }
        return realpath(ee()->config->item('base_path')) . '/';
    }

    /**
     * Gets information from a user agent string
     * from https://gist.github.com/james2doyle/5774516
     *
     * @param  string $u_agent
     * @return array
     */
    public function getBrowser($u_agent = null)
    {

        $bname = null;
        $platform = null;
        $version = null;
        $ub = '';

        // First get the platform?
        if (preg_match('/linux/i', $u_agent)) {
            $platform = 'linux';
        } elseif (preg_match('/macintosh|mac os x/i', $u_agent)) {
            $platform = 'mac';
        } elseif (preg_match('/windows|win32/i', $u_agent)) {
            $platform = 'windows';
        }

        // Next get the name of the useragent yes seperately and for good reason
        if (preg_match('/MSIE/i', $u_agent) && !preg_match('/Opera/i', $u_agent)) {
            $bname = 'Internet Explorer';
            $ub = "MSIE";
        } elseif (preg_match('/Firefox/i', $u_agent)) {
            $bname = 'Mozilla Firefox';
            $ub = "Firefox";
        } elseif (preg_match('/Chrome/i', $u_agent)) {
            $bname = 'Google Chrome';
            $ub = "Chrome";
        } elseif (preg_match('/Safari/i', $u_agent)) {
            $bname = 'Apple Safari';
            $ub = "Safari";
        } elseif (preg_match('/Opera/i', $u_agent)) {
            $bname = 'Opera';
            $ub = "Opera";
        } elseif (preg_match('/Netscape/i', $u_agent)) {
            $bname = 'Netscape';
            $ub = "Netscape";
        }

        // finally get the correct version number
        $known = array('Version', $ub, 'other');
        $pattern = '#(?<browser>' . join('|', $known) . ')[/ ]+(?<version>[0-9.|a-zA-Z.]*)#';
        if (!preg_match_all($pattern, $u_agent, $matches)) {
            // we have no matching number just continue
        }

        // see how many we have
        $i = count($matches['browser']);
        if ($i != 1) {
            //we will have two since we are not using 'other' argument yet
            //see if version is before or after the name
            if (strripos($u_agent, "Version") < strripos($u_agent, $ub)) {
                $version = $matches['version'][0];
            } else {
                $version = $matches['version'][1];
            }
        } else {
            $version = $matches['version'][0];
        }

        // check if we have a number
        if ($version == null || $version == "") {
            $version = "?";
        }

        return array(
            'userAgent' => $u_agent,
            'name'      => $bname,
            'version'   => $version,
            'platform'  => $platform,
            'pattern'    => $pattern
        );
    }

    /**
     * JCOGS EE Auto-Translate - getCacheKey utility 
     * =============================================
     * Builds a context sensitive cache key to use with EE's cache class
     *
     * @param  string $unique_id
     * @return string
     */
    public function getCacheKey($unique_id)
    {
        // $class = __CLASS__ ? __CLASS__.'/' : '';
        return '/' . JCOGS_IMG_CLASS . '/' . $unique_id;
    }

    /**
     * Utility function: Get a remote file using CURL
     * Returns either the file or false
     *
     * @param string $path
     * @return object|bool
     */
    public function get_file_contents_curl(string $path)
    {
        if (!$path) {
            // No path
            $this->debug_message(lang('jcogs_utils_gfcc_no_path'), $path);
            return false;
        }
        if (!function_exists('curl_init')) {
            // No CURL
            $this->debug_message(lang('jcogs_utils_gfcc_no_curl'), $path);
            return false;
        }

        // Set up CURL transaction

        // Thanks to ... https://stackanswers.net/questions/file-get-contents-fails-via-php-works-via-browser
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_URL, $path);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        // curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);

        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 2);
        curl_setopt($ch, CURLOPT_TIMEOUT, 5);
        curl_setopt($ch, CURLOPT_USERAGENT, $this->settings['img_cp_default_user_agent_string']);
        // curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 6.1; WOW64; rv:5.0) Gecko/20100101');

        // if either of these options are enabled, need this to skip an error
        if (!ini_get('open_basedir') && !ini_get('safe_mode')) {
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        }

        // Get the image if you can
        $remote_file = curl_exec($ch);
        curl_close($ch);

        $remote_file ? $this->debug_message(lang('jcogs_utils_gfcc_success'), $path) : $this->debug_message(lang('jcogs_utils_gfcc_failed'), $path);

        return $remote_file;
    }

    /**
     * Utility function: Get a file from remote source
     * Uses file_get_contents, but if that fails tries two alternatives:
     * - less secure file_get_contents
     * - CURL
     * Returns the file or false
     *
     * @param string $path
     * @return object|bool
     */
    public function get_file_from_remote(string $path)
    {
        // No point trying if we can't access remote files... 
        if (!$this->allow_url_fopen_enabled()) {
            $this->debug_message(lang('jcogs_utils_allow_url_fopen_disabled'));
            return false;
        }

        $this->debug_message(lang('jcogs_utils_gffr_started'), $path);
        // Try first using file_get_contents, which works most of the time

        // Do some connection massaging... 
        $options = array(
            'http' => array(
                'method' => "GET",
                'header' => "Accept-language: en\r\n",
                'timeout' => $this->settings['img_cp_default_php_remote_connect_time'],
                'user_agent' => $this->settings['img_cp_default_user_agent_string']
            )
        );
        $context = stream_context_create($options);

        // Do some housekeeping to encourage file transfer can happen on difficult hosts
        @ini_set('allow_url_fopen', true);

        $remote_file = false;
        $remote_path_array = pathinfo($path);
        $clean_remote_path = $remote_path_array['dirname'] . '/' . urlencode($remote_path_array['filename']);
        $clean_remote_path .= isset($remote_path_array['extension']) ? '.' . $remote_path_array['extension'] : '';
        $remote_file = @file_get_contents($clean_remote_path, false, $context);
        if (!$remote_file) {
            // No joy? 
            $this->debug_message(lang('jcogs_utils_unable_to_open_path_4'), $path);
            // Have a final go using CURL ...
            $remote_file = $this->get_file_contents_curl($path);
            if (!$remote_file) {
                // unable to read file from remote location
                return false;
            }
        }
        // $this->debug_message(lang('jcogs_utils_gffr_ending'),$path);
        return $remote_file;
    }

    /**
     * JCOGS EE Auto-Translate - obscure_key utility 
     * =============================================
     * Returns the first n characters of a string
     *
     * @param  string $key
     * @return string $key
     */
    public function obscure_key(string &$key, $length = 10)
    {
        if (!is_string($key) || strlen($key) <= $length) {
            return '';
        }
        return substr($key, 0, $length) . str_repeat('•', strlen($key) - $length);
    }

    /**
     * Utility function: Output JSON and die
     *
     * @param string $rel_path
     * @param boolean $mk_dir
     * @return string
     */
    public function output_json($json)
    {
        header('Content-type: application/json');
        echo json_encode($json);
        die;
    }

    /**
     * Convert {filedir_X} to real path
     * 
     * @param string $location
     * @return string|bool
     */
    public function parseFiledir($location)
    {
        $path = '';
        if (substr(APP_VER, 0, 1) == 7) {
            ee()->load->library('file_field');
            $file_path = ee()->file_field->getFileModelForFieldData($location);
            if ($file_path) {
                // If location given is not valid filedir we get nothing, so don't do any more unless we do
                $path = ee()->file_field->getFileModelForFieldData($location)->getAbsolutePath();
                // string returned by getAbsolutePath() will include the basepath, so for compatibility we need to remove this
                // first check that $path does indeed contain the base_path
                if (str_contains($path, $this->get_base_path()) === false) {
                    // weird stuff going on so bale... 
                    $this->debug_message(lang('jcogs_utils_no_base_path'), [$path]);
                    return false;
                } else {
                    $path = str_replace(rtrim($this->get_base_path(), '/'), '', $path);
                }
            } else {
                // Location is not a file path so return empty string
                return '';
            }
        } else {
            // Do we have a string or a number ... ?
            if(str_contains($location,'filedir_')) {
                preg_match('/{filedir_(.*?)}(.*)$/', $location, $matches);
                if (isset($matches[0])) {
                    $location = $matches[1];
                } else {
                    return '';
                }
            }
            if (intval($location) > 0) {
                $path = ee('Model')->get('UploadDestination')->filter('id', $location)->first()->url;
                $path = isset($matches[2]) ? $path . $matches[2] : $path;
            }
        }
        return $path;
    }

    /**
     * Utility function: Optionally creates and returns the path in which we will be working with
     * our files
     *
     * @param string $rel_path
     * @param boolean $mk_dir
     * @return string
     */
    public function path($rel_path = '', $mk_dir = false)
    {
        // Get basepath, add rel path and check if exists.
        if (!$path = $this->get_base_path()) {
            // We cannot operate without a valid base_path so bale out!
            $this->debug_message(lang('jcogs_utils_no_base_path'), [$path]);
            return false;
        };

        // Check for and remove double-slashes if any present in composite path
        ee()->load->helper('string');
        $clean_path = reduce_double_slashes($path . $rel_path);

        // Got a good base_path so test the rest of the path provided ... 
        if (!ee('Filesystem')->exists($clean_path) && $mk_dir) {
            ee('Filesystem')->mkDir($clean_path);
        }
        return rtrim($clean_path, '/') . '/';
    }

    /**
     * Checks current php version for being > 5.4...
     *
     * @return bool|int
     */
    public function valid_ee_version()
    {
        return version_compare(APP_VER, '5.4.0', '>=');
    }

    /**
     * Checks current php version for being > 7.4...
     *
     * @return bool|int
     */
    public function valid_php_version()
    {
        return version_compare(PHP_VERSION, '7.4.0', '>=');
    }

    /**
     * Utility function: check font size value and normalise to px
     * 
     * Uses px = 72/96 * pt principle from here - https://pixelsconverter.com/px-to-pt
     * 
     * @param string $font_size_string
     * @return string $font_size_px
     */
    public function validate_font_size($font_size_string)
    {
        // Check to see if the string has px or pt endings and modify value accordingly
        if (stripos($font_size_string, 'pt')) {
            $font_size = str_replace('pt', '', strtolower($font_size_string));
            return (int) ($font_size * 72 / 96);
        }
        if (stripos($font_size_string, 'px')) {
            $font_size_string = str_replace('px', '', strtolower($font_size_string));
        }
        return (int)($font_size_string);
    }

    /**
     * If a namespace was specified, prefixes the key with it
     *
     * For the file driver, namespaces will be actual folders
     *
     * @param	string	$key	Key name
     * @param	mixed	$scope	Cache::LOCAL_SCOPE or Cache::GLOBAL_SCOPE
     *		 for local or global scoping of the cache item
     * @return	string	Key prefixed with namespace
     */
    protected function _namespaced_key($key, $scope = \Cache::LOCAL_SCOPE)
    {
        // Make sure the key doesn't begin or end with a namespace separator or
        // directory separator to force the last segment of the key to be the
        // file name and so we can prefix a directory reliably
        $key = trim($key, \Cache::NAMESPACE_SEPARATOR . DIRECTORY_SEPARATOR);

        // Sometime class names are used as keys, replace class namespace
        // slashes with underscore to prevent filesystem issues
        $key = str_replace('\\', '_', $key);

        // Replace all namespace separators with the system's directory separator
        $key = str_replace(\Cache::NAMESPACE_SEPARATOR, DIRECTORY_SEPARATOR, $key);

        // For locally-cached items, separate by site name
        if ($scope == \Cache::LOCAL_SCOPE) {
            $key = (!empty(ee()->config->item('site_short_name')) ? ee()->config->item('site_short_name') . DIRECTORY_SEPARATOR : '') . $key;
        }

        return $key;
    }
}
