<?php if (!defined('BASEPATH')) exit('No direct script access allowed');
/**
 * Image Module
 * ============
 * Full function image processing add-on for EE.
 * Designed to work with similar parameter and variable definitions
 * to CE-Image, to support drop-in replacement.
 *                      
 * =====================================================
 *
 * @category   ExpressionEngine Add-on
 * @package    JCOGS Image
 * @author     JCOGS Design <contact@jcogs.net>
 * @copyright  Copyright (c) 2021 - 2023 JCOGS Design
 * @license    https://jcogs.net/add-ons/jcogs_img/license.html
 * @version    1.3.21.1
 * @link       https://JCOGS.net/
 * @since      File available since Release 1.0.0
 */

require_once PATH_THIRD . "jcogs_img/vendor/autoload.php";
require_once PATH_THIRD . "jcogs_img/config.php";

use JCOGSDesign\Jcogs_img\Library\JcogsImage;
use Imagine\Gd\Imagine;
use Imagine\Image\PointSigned;
use ColorThief\ColorThief;

class Jcogs_img
{
    private $has_started = false;
    private $EE;
    private $setupIsValid;
    public $settings;

    function __construct()
    {
        $this->EE = get_instance();
        $this->settings = ee('jcogs_img:Settings')::$settings;
        $this->setupIsValid = true;

        // Check we have a license in place
        if (!($this->settings['jcogs_license_mode'] == 'valid' || $this->settings['jcogs_license_mode'] == 'magic' || $this->settings['jcogs_license_mode'] == 'staging')) {
            // Check to see if we can run in demo mode
            // From 1.2.8 onwards we can always run in demo mode if no license set
            // So put a marker in the log to note demo mode operational
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_demo_mode'));
        }

        // Check php version is valid... 
        if (!ee('jcogs_img:Utilities')->valid_php_version()) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_invalid_php_version'));
            $this->setupIsValid = false;
        }

        // Check EE version is valid... 
        if (!ee('jcogs_img:Utilities')->valid_ee_version()) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_invalid_ee_version'));
            $this->setupIsValid = false;
        }

        // Check if we have a GD image library instance to work with...         
        if (!extension_loaded('gd')) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_gd_library_not_found'));
            $this->setupIsValid = false;
        }

        // Does this EE installation have a base_path set? If not, bale (as we don't know 
        // where to write the image files we create!
        if (!ee('Filesystem')->exists(ee()->config->item('base_path'))) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_no_base_path'));
            $this->setupIsValid = false;
        }

        // Add a hook for CE Img quasi-compatibility
        if ($this->setupIsValid && !$this->has_started) {
            $this->has_started = true;
            if (ee()->extensions->active_hook('jcogs_img_start')) {
                ee()->extensions->call('jcogs_img_start');
            }
        }
    }

    /**
     * Shell function that finds <img> tags in a block of tag data and submits
     * image() calls for each
     *
     * @return mixed
     */
    public function bulk()
    {
        // We can only start if the session is a valid one
        if (!$this->setupIsValid) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_setup_is_invalid'));
            return false;
        }

        $vars = []; // This is container for the processed images we are going to generate

        $bulk_params = new stdClass;
        // Get whatever this->content / parameters have been provided in the tag
        $bulk_params->exclude_regex = ee('jcogs_img:ImageUtilities')->get_parameters('exclude_regex');

        // Grab the tagdata and fish out the <img> tags (if any)
        preg_match_all('/(:?<img\s(.*?)>)/s', ee()->TMPL->tagdata, $img_tags, PREG_SET_ORDER);

        // Exclude any images that match the regex provided in the exclude_regex parameter
        if ($bulk_params->exclude_regex) {
            $regexs = explode('@', $bulk_params->exclude_regex);
            foreach ($regexs as $regex) {
                $i = 0;
                foreach ($img_tags as $img_tag) {
                    if (preg_match('/' . $regex . '/', $img_tag[2])) {
                        ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_bulk_excluding_image'), $img_tag[2]);
                        unset($img_tags[$i]);
                    }
                    $i++;
                }
            }
        }

        // Did we get anything to process ... ?
        if (!$img_tags) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_bulk_processing_no_images_found'));
            return ee()->TMPL->tagdata;
        }

        // Build a text list of images we are going to work on for debug message:
        $image_list = [];
        foreach ($img_tags as $img_tag) {
            array_push($image_list, $img_tag[2]);
        }

        ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_bulk_processing_start'), $image_list);

        // Process each <img> tag found
        $i = 0;
        foreach ($img_tags as $img_tag) {
            if (is_null($img_tag) || $img_tag == '') {
                // no content in img tag found so bale
                return;
            }

            // To avoid filter confusion, get the bulk tag params again
            $bulk_params = ee('jcogs_img:ImageUtilities')->get_parameters();
            // Force output of an <img> tag 
            $bulk_params->bulk_tag = 'y';

            // Clear the $image_tag object if it exists to prevent carry-over of properties etc.
            if (isset($image_tag)) unset($image_tag);

            // Create a container to submit image_process call with
            $image_tag = new JcogsImage;

            // Start with the bulk parameters
            $image_tag->params = $bulk_params;

            // Reset attributes parameter
            // $image_tag->params->attributes = '';

            $temp_var = 'new_img_tag_' . $i;
            // replace found tag with a marker
            ee()->TMPL->tagdata = str_replace($img_tag[1], '{' . $temp_var . '}', ee()->TMPL->tagdata);

            // put current content of tag found into vars as default value
            $vars[0]['new_img_tag_' . $i] = $img_tag[1];

            // For each tag fish out the src, width and height parameters and remove from source
            // Get src...
            if (!preg_match('/(?:src=(?:\"|\')(.*?)(?:\"|\'))/', $img_tag[2], $src)) {
                // no src found so bale
                ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_bulk_no_source'), $img_tag[2]);
                return;
            }
            // remove the src from $img_tag[2]
            $img_tag[2] = str_replace($src[0], '', $img_tag[2]);
            // check to see if this is a lazy-load src and swap in real image if so
            if (stripos(strtolower($src[1]), '_lqip_')) {
                $src[1] = str_replace('lqip_', '', $src[1]);
                ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_bulk_found_lazy_swap'), $src[1]);
            } elseif (stripos(strtolower($src[1]), '_dominant_color_')) {
                $src[1] = str_replace('dominant_color_', '', $src[1]);
                ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_bulk_found_lazy_swap'), $src[1]);
            }
            // add / update params with the value
            $image_tag->params->src = $src[1];

            // Get width... and if found, remove it from $img_tag[]
            if (trim($img_tag[2]) != '' && preg_match('/(?:width=(?:\"|\')(.*?)(?:\"|\'))/', $img_tag[2], $width)) {
                // remove the value from $img_tag[2]
                $img_tag[2] = str_replace($width[0], '', $img_tag[2]);
                // add / update params with the value only if no over-ruling bulk tag param set
                $image_tag->params->width = $image_tag->params->width ?: $width[1];
            };

            // Get height... and if found, remove it from $img_tag[]
            if (trim($img_tag[2]) != '' && preg_match('/(?:height=(?:\"|\')(.*?)(?:\"|\'))/', $img_tag[2], $height)) {
                // remove the value from $img_tag[2]
                $img_tag[2] = str_replace($height[0], '', $img_tag[2]);
                // add / update params with the value
                $image_tag->params->height = $image_tag->params->height ?: $height[1];
            };

            // If anything left in tag put that into the attributes parameter
            if (trim($img_tag[2]) != '') {
                $image_tag->params->attributes = property_exists($image_tag->params, 'attributes') ? trim($image_tag->params->attributes . ' ' . $img_tag[2]) : trim($img_tag[2]);
            }

            // Process the image 
            $vars[0][$temp_var] = $this->image($image_tag);
            $i++;
        }

        // Process the output to give updated tagdata

        return ee()->TMPL->parse_variables(ee()->TMPL->tagdata, $vars);
    }

    /**
     * Shell function to provide 'single' tag option
     *
     * @return object
     */
    public function single()
    {
        return $this->image();
    }

    /**
     * Shell function to provide 'size' tag option
     *
     * @return object
     */
    public function size()
    {
        return $this->image();
    }

    /**
     * Shell function to provide 'pair' tag option
     *
     * @return object
     */
    public function pair()
    {
        return $this->image();
    }

    /**
     * Function to provide 'palette' tag option
     * Still need to process image, so this uses standard image call
     * and then adds call to palette method before returning palette
     *
     * @return mixed
     */
    public function palette()
    {
        // We can only start if the session is a valid one
        if (!$this->setupIsValid) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_setup_is_invalid'));
            return false;
        }

        if (!$content = $this->image(null, true)) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_palette_no_image'));
            return;
        }

        // If using cache copy try creating image
        if ($content->flags->using_cache_copy) {
            try {
                $content->processed_image = (new Imagine())->open($content->save_path);
            } catch (\Imagine\Exception\RuntimeException $e) {
                // Creation of image failed.
                ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_imagine_error'), $e->getMessage());
                return;
            }
        }

        $palette = ColorThief::getPalette($content->processed_image->getGdResource(), $content->params->palette_size, 10, null, 'rgb');

        $dc = ColorThief::getColor($content->processed_image->getGdResource(), 10, null, 'rgb');

        $i = 0;
        foreach ($palette as $color) {
            $output[0]['colors'][$i] = ['color' => $color, 'rank' => $i + 1];
            $i++;
        }

        $output[0]['dominant_color'] = $dc;

        if (empty($palette)) {
            unset($content);
            return ee()->TMPL->no_results();
        }

        unset($raw_palette);
        unset($palette);
        unset($content);
        return ee()->TMPL->parse_variables(ee()->TMPL->tagdata, $output);
    }

    /**
     * Primary method for producing image results back to EE template
     * The production process is split into four parts:
     * 1) Setup - work out name for processed and check if it is in cache,
     * 2) Process - If not in cache, validate and process image
     * 3) Transform - Apply whole-image transforms (filters, border, adding text/watermark)
     * 4) Save - Save image
     * 5) Post-process - Assemble parsed variables, generate template output requested
     *
     * @return bool|object $content
     */
    public function image($content = null, $return_content = false)
    {
        // We can only start if the session is a valid one
        if (!$this->setupIsValid) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_setup_is_invalid'));
            return false;
        }

        // 1: Do some housekeeping and setup local copy of image to work with
        // ==================================================================

        // Start a timer for this execution run
        $time_start = microtime(true);

        // Request minimum amount of memory for php process ... 
        // 1 - what is it currently?
        $current_php_memory = str_replace('M', '', ini_get('memory_limit'));

        // 2 - if it is less than default temporarily ask for default amount
        if ($current_php_memory < $this->settings['img_cp_default_min_php_ram']) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_memory_uplift'), $current_php_memory . 'M / ' . $this->settings['img_cp_default_min_php_ram'] . 'M');
            @ini_set('memory_limit', $this->settings['img_cp_default_min_php_ram'] . 'M');
        }

        // Request minimum execution time for php process ... 
        // 1 - what is it currently?
        $current_php_execution_time_limit = ini_get('max_execution_time');
        // 2 - if it is less than default temporarily ask for default amount
        if ($current_php_execution_time_limit < $this->settings['img_cp_default_min_php_process_time']) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_process_time_uplift'), $current_php_execution_time_limit . 's / ' . $this->settings['img_cp_default_min_php_process_time'] . 's');
            @ini_set('max_execution_time', $this->settings['img_cp_default_min_php_process_time']);
        }

        // Do a cache audit if one is due
        ee('jcogs_img:ImageUtilities')->cache_audit();

        // If we are not doing bulk processing and no passed content create a new JcogsImage object
        if (!is_object($content)) {
            $content = new JcogsImage;
        }

        // Initialise the image
        if (!$content->initialise()) {
            // If we get here something went wrong
            return false;
        }

        // 2: Process the image with parameters given
        // ==========================================

        if (!$content->flags->using_cache_copy && !$content->process_image()) {
            // Something went wrong!
            return false;
        }

        // 3: If we are in demo mode overlay demo watermark
        // ================================================
        // We do this here rather than in Image class so that only one demo label applied even
        // if image is a composite

        // If image is a cache copy, an svg or an animated gif - skip rest of this step
        // if (!$content->flags->using_cache_copy && !$content->flags->svg && !($content->flags->animated_gif && $content->params->save_as != 'gif')) {
        //     if (!$content->flags->using_cache_copy && $this->settings['jcogs_license_mode'] == 'demo') {
        //         // Get demo image
        //         $demo_image =  (new Imagine())->load(base64_decode(ee('jcogs_img:ImageUtilities')->demo_image()));
        //         // Get size of demo image
        //         $demo_image_size = $demo_image->getSize();
        //         // Paste demo image onto working image
        //         $content->processed_image->paste($demo_image, new PointSigned(round(($content->new_width - $demo_image_size->getWidth()) / 2, 0), round(($content->new_height - $demo_image_size->getHeight()) / 2, 0)));
        //     }
        // }

        // 4: Save - write out image to disk
        // =================================

        if (!$content->flags->using_cache_copy && !$content->save()) {
            return false;
        }

        // 5: Post-process - generate parsed variables
        // ===========================================

        if (!$content->post_process()) {
            return false;
        }

        // 6) Generate output... 
        // ===========================================
        if (!$content->generate_output()) {
            return false;
        }

        // 7) Finish - clean up and return output
        // ======================================

        // Get the finish time for processing
        $elapsed_time = microtime(true) - $time_start;
        if ($elapsed_time > 2) {
            $elapsed_time_report =  '<span style="color:var(--ee-error-dark);font-weight:bold">Processing time: %0.4f seconds</span>';
        } elseif ($elapsed_time > 1) {
            $elapsed_time_report =  '<span style="color:var(--ee-warning-dark);font-weight:bold">Processing time: %0.4f seconds</span>';
        } else {
            $elapsed_time_report =  '<span style="color:var(--ee-button-success-hover-bg);font-weight:bold">Processing time: %0.4f seconds</span>';
        }

        ee('jcogs_img:Utilities')->debug_message(sprintf(lang('jcogs_img_processing_end'), sprintf($elapsed_time_report, $elapsed_time)));

        // Restore original php memory and execution time limits
        if ($current_php_memory !=  str_replace('M', '', ini_get('memory_limit'))) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_restore_memory'), $current_php_memory . 'M');
            @ini_set('memory_limit', $current_php_memory . 'M');
        }

        // Restore original minimum execution time for php process ... 
        if ($current_php_execution_time_limit != ini_get('max_execution_time')) {
            ee('jcogs_img:Utilities')->debug_message(lang('jcogs_img_restore_process_time'), $current_php_execution_time_limit . 's');
            @ini_set('max_execution_time', $current_php_execution_time_limit);
        }

        if ($return_content) {
            // return $content to somewhere else for further processing
            return $content;
        } else {
            // we are done here so exit
            $what_to_return = $content->return;
            unset($content);
            return $what_to_return;
        }
    }
}
