<?php
/**
 * Redirect Pro
 *
 * @package     ExpressionEngine
 * @subpackage  Addons
 * @category    Module
 * @author      Imam Husayn
 * @link        
 */

use EllisLab\ExpressionEngine\Library\CP\Table;

include_once 'addon.setup.php';

Class Redirect_pro_mcp{

	protected $module;
    protected $base_url;
    protected $make_site_url;
    protected $date_format;
    protected $date_picker_format;

    public function __construct()
    {
        $this->module = IHRP_MODULE_NAME;
        $this->base_url = ee('CP/URL', 'addons/settings/' . $this->module);
        
        //get current site URL
        $site_url   = (ee()->config->item('site_url')) ? rtrim(ee()->config->item('site_url'),'/') . '/' : '';
        $site_index = (ee()->config->item('site_index')) ? rtrim(ee()->config->item('site_index'),'/') .'/' : '';
        $this->make_site_url = $site_url . $site_index;

        //get date format
        $this->date_picker_format = ee()->config->item('date_format');
        $this->date_format = str_replace('%', '', ee()->config->item('date_format'));

        //make sidebar
        $this->sidebar = ee('CP/Sidebar')->make();

        $this->items_nav = $this->sidebar->addHeader(lang('Redirects'), ee('CP/URL')->make('addons/settings/redirect_pro'))
            ->withButton(lang('new'), ee('CP/URL')->make('addons/settings/redirect_pro/create'));

    }

	public function index(){

		$table = ee('CP/Table');
        $table->setColumns(array(
            'id',
            'from',
            'to',
            'start_date',
            'end_date',
            'count',
            'manage' => array(
                'type' => Table::COL_TOOLBAR
            ),
            array(
                'type' => Table::COL_CHECKBOX
            )
        ));

        $table->setNoResultsText(sprintf(lang('no_found'), lang('redirects')), 'create_redirect', ee('CP/URL')->make('addons/settings/redirect_pro/create'));

        $sort_map = array(
            'id' => 'redirect_id',
            'from' => 'redirect_from',
            'to' => 'redirect_to',
            'start_date' => 'start_date',
            'end_date' => 'end_date',
            'count' => 'count',
        );

        $redirects = ee()->db->select('*')
            ->order_by($sort_map[$table->sort_col], $table->sort_dir)
            ->get('redirects')
            ->result_array();

        $data = array();
        foreach ($redirects as $redirect) {

            $redirect_from = $this->make_site_url.ltrim($redirect['redirect_from'], '/');

            $edit_url = ee('CP/URL')->make('addons/settings/redirect_pro/edit/' . $redirect['redirect_id']);

            $columns = array(
                $redirect['redirect_id'],
                array(
                    'content' => $redirect['redirect_from'],
                    'href' => $edit_url
                ),
                array(
                    'content' => $redirect['redirect_to'],
                ),
                array(
                    'content' => empty($redirect['start_date']) ? '' : date($this->date_format, strtotime($redirect['start_date'])),
                ),
                array(
                    'content' => empty($redirect['end_date']) ? '' : date($this->date_format, strtotime($redirect['end_date'])),
                ),
                array(
                    'content' => $redirect['count'],
                ),
                array('toolbar_items' => array(
                    'edit' => array(
                        'href' => $edit_url,
                        'title' => lang('edit')
                    ),
                    'txt-only' => array(
                        'href' => $redirect_from,
                        'title' => (lang('check_redirect')),
                        'content' => strtolower(lang('check_redirect')),
                        'target' => '_blank'
                    )
                )),
                array(
                    'name' => 'redirects[]',
                    'value' => $redirect['redirect_id'],
                    'data' => array(
                        'confirm' => lang('redirect') . ': <b>' . htmlentities($redirect['redirect_from'], ENT_QUOTES, 'UTF-8') . '</b>'
                    )
                )
            );

            $attrs = array();
            if (ee()->session->flashdata('highlight_id') == $redirect['redirect_id']) {
                $attrs = array('class' => 'selected');
            }

            $data[] = array(
                'attrs' => $attrs,
                'columns' => $columns
            );
        }

        $table->setData($data);

        $vars['base_url'] = ee('CP/URL')->make('addons/settings/redirect_pro');
        $vars['table'] = $table->viewData($vars['base_url']);


        //provide pagination next update
        $vars['pagination'] = ee('CP/Pagination', count($redirects))
            /*->perPage($vars['table']['limit'])
            ->currentPage($vars['table']['page'])*/
            ->render($vars['table']['base_url']);

        ee()->javascript->set_global('lang.remove_confirm', lang('redirects') . ': <b>### ' . lang('redirects') . '</b>');
        ee()->cp->add_js_script(array(
            'file' => array('cp/confirm_remove'),
        ));

        return array(
            'heading' => lang('redirect_pro'),
            'body' => ee('View')->make('redirect_pro:listing')->render($vars),
            'sidebar' => $this->sidebar
        );

	}

    public function create(){
        return $this->create_update();
    }

    public function edit($redirect_id){
        return $this->create_update($redirect_id);
    }

    public function remove(){

        $redirect_ids = ee()->input->post('redirects');

        if (! empty($redirect_ids) && ee()->input->post('bulk_action') == 'remove') {

            $redirect_ids = array_filter($redirect_ids, 'is_numeric');

            if (! empty($redirect_ids)) {
                ee('Model')->get('redirect_pro:Redirect', $redirect_ids)->delete();

                ee('CP/Alert')->makeInline('redirects-table')
                    ->asSuccess()
                    ->withTitle(lang('redirects_removed'))
                    ->addToBody(sprintf(lang('redirects_removed_desc'), count($redirect_ids)))
                    ->defer();
            }
        } else {
            show_error(lang('unauthorized_access'), 403);
        }

        ee()->functions->redirect(ee('CP/URL')->make('addons/settings/redirect_pro', ee()->cp->get_url_state()));
    }

    private function create_update($redirect_id = ''){

        $vars = array();
        if (empty($redirect_id)) {

            $alert_key = 'created';
            $vars['cp_page_title'] = lang('create_redirect');
            $vars['base_url'] = ee('CP/URL')->make('addons/settings/redirect_pro/create');

            $redirect = ee('Model')->make('redirect_pro:Redirect');
        } else {
            $redirect = ee('Model')->get('redirect_pro:Redirect', $redirect_id)->first();

            if (! $redirect) {
                show_error(lang('unauthorized_access'), 403);
            }

            $alert_key = 'updated';
            $vars['cp_page_title'] = lang('edit_redirect');
            $vars['base_url'] = ee('CP/URL')->make('addons/settings/redirect_pro/edit/' . $redirect_id);
        }

        if (!empty($_POST)) {

            /*print_r($_POST);
            exit;*/
            $redirect->set($_POST);

            $result = $redirect->validate();

            if ($result->isValid()) {
                
                $redirect->site_id = ee()->config->item('site_id');   

                $redirect->redirect_from = str_replace($this->make_site_url, '', $redirect->redirect_from);
                $redirect->redirect_to = str_replace($this->make_site_url, '', $redirect->redirect_to);

                $redirect->start_date = ( isset($redirect->start_date) && !empty($redirect->start_date) ) ? date('Y-m-d', strtotime($redirect->start_date)) : null;

                $redirect->end_date = (isset($redirect->end_date) && !empty($redirect->end_date)) ? date('Y-m-d', strtotime($redirect->end_date)) : null;

                $redirect = $redirect->save();

                if (empty($redirect_id)) {
                    ee()->session->set_flashdata('highlight_id', $redirect->getId());
                }

                ee('CP/Alert')->makeInline('redirects-table')
                    ->asSuccess()
                    ->withTitle(lang('redirect_' . $alert_key))
                    ->addToBody(sprintf(lang('redirect_' . $alert_key . '_desc'), $redirect->redirect_from))
                    ->defer();

                ee()->functions->redirect(ee('CP/URL')->make('addons/settings/redirect_pro'));
            } else {
                $vars['errors'] = $result;
                ee('CP/Alert')->makeInline('redirects-table')
                    ->asIssue()
                    ->withTitle(lang('redirect_not_' . $alert_key))
                    ->addToBody(lang('redirect_not_' . $alert_key . '_desc'))
                    ->now();
            }
        }

        $vars['sections'] = array(
            array(
                array(
                    'title' => 'from',
                    'desc' => 'Enter the website path that you want to redirect from it. Example: path/old-about <br/> Notes: Don\'t write to add the complete website URL.',
                    'fields' => array(
                        'redirect_from' => array(
                            'type' => 'text',
                            'value' => $redirect->redirect_from,
                            'required' => true
                        )
                    )
                ),
                array(
                    'title' => 'to',
                    'desc' => 'Enter the your website path that you want to redirect to it. Example: path/to/new-about  <br/> You can also add an enternal URL. Example: https://www.google.com ',
                    'fields' => array(
                        'redirect_to' => array(
                            'type' => 'text',
                            'value' => $redirect->redirect_to,
                            'required' => true
                        )
                    )
                ),
                array(
                    'title' => 'start_date',
                    'desc' => 'The date when this redirect will be start. <i>(Optional)</i>',
                    'fields' => array(
                        'start_date' => array(
                            'type' => 'text',
                            'value' => empty($redirect->start_date) ? '' : date($this->date_format, strtotime($redirect->start_date)),
                            'required' => false,
                            'attrs' => 'autocomplete="off" rel="date-picker" data-timestamp="' . strtotime($redirect->start_date) . '"'
                        )
                    )
                ),
                array(
                    'title' => 'end_date',
                    'desc'  => 'The date when this redirect will be stop. <i>(Optional)</i>',
                    'fields' => array(
                        'end_date' => array(
                            'type' => 'text',
                            'value' => empty($redirect->end_date) ? '' : date($this->date_format, strtotime($redirect->end_date)),
                            'required' => false,
                            'attrs' => 'autocomplete="off" rel="date-picker" data-timestamp="' . strtotime($redirect->end_date) . '"'
                        )
                    )
                )
            )
        );

        ee()->javascript->compile();

        $vars['save_btn_text'] = 'save_redirect';
        $vars['save_btn_text_working'] = 'btn_saving';

        ee()->javascript->set_global('date.date_format', $this->date_picker_format);
        ee()->javascript->set_global('lang.date.months.full', array(
            lang('january'),
            lang('february'),
            lang('march'),
            lang('april'),
            lang('may'),
            lang('june'),
            lang('july'),
            lang('august'),
            lang('september'),
            lang('october'),
            lang('november'),
            lang('december')
        ));
        ee()->javascript->set_global('lang.date.months.abbreviated', array(
            lang('jan'),
            lang('feb'),
            lang('mar'),
            lang('apr'),
            lang('may'),
            lang('june'),
            lang('july'),
            lang('aug'),
            lang('sept'),
            lang('oct'),
            lang('nov'),
            lang('dec')
        ));
        ee()->javascript->set_global('lang.date.days', array(
            lang('su'),
            lang('mo'),
            lang('tu'),
            lang('we'),
            lang('th'),
            lang('fr'),
            lang('sa'),
        ));
        ee()->cp->add_js_script(array(
            'file' => array('cp/date_picker'),
        ));

        return array(
            'heading' => $vars['cp_page_title'],
            'breadcrumb' => array(ee('CP/URL')->make('addons/settings/redirect_pro')->compile() => lang('redirects')),
            'body' => ee('View')->make('redirect_pro:create')->render($vars),
            'sidebar' => $this->sidebar
        );

    }

}